# Chat System Implementation

## Overview
This document describes the implementation of a real-time chat and collaboration system for the Hybrid ERP platform. The system enables users to communicate, share information, and collaborate through both direct messages and group conversations.

## Features

### Core Functionality
- **Real-time Messaging**: Instant message delivery using WebSocket connections
- **Conversation Management**: Create and manage group and direct conversations
- **User Authentication**: Secure WebSocket connections with JWT authentication
- **Message History**: Persistent storage and retrieval of chat messages
- **Unread Message Tracking**: Track unread messages per conversation
- **Typing Indicators**: Real-time typing status for better user experience

### User Experience
- **Modern UI**: Clean, responsive interface built with Quasar Framework
- **Conversation List**: Easy navigation between different chat threads
- **Search Functionality**: Find conversations and messages quickly
- **Real-time Updates**: Live message delivery and conversation updates
- **Mobile Responsive**: Works seamlessly across all device sizes

## Architecture

### Backend Components

#### 1. Database Schema
```sql
-- Conversations table
CREATE TABLE conversations (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(255) NULL,
    type ENUM('direct', 'group') DEFAULT 'group',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Conversation participants
CREATE TABLE conversation_participants (
    id INT PRIMARY KEY AUTO_INCREMENT,
    conversation_id INT NOT NULL,
    user_id INT NOT NULL,
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_read_at TIMESTAMP NULL,
    FOREIGN KEY (conversation_id) REFERENCES conversations(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Messages table
CREATE TABLE messages (
    id INT PRIMARY KEY AUTO_INCREMENT,
    conversation_id INT NOT NULL,
    sender_id INT NOT NULL,
    message TEXT NOT NULL,
    message_type ENUM('text', 'file', 'image', 'system') DEFAULT 'text',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

#### 2. API Endpoints
- `GET /chat/conversations` - Get user's conversations
- `POST /chat/conversations` - Create new conversation
- `GET /chat/conversations/:id/messages` - Get conversation messages
- `GET /chat/conversations/:id/participants` - Get conversation participants
- `POST /chat/messages` - Send message (REST fallback)
- `GET /chat/conversations/:id/unread` - Get unread message count

#### 3. WebSocket Events
- `authenticate` - Authenticate socket connection
- `join_conversation` - Join conversation room
- `leave_conversation` - Leave conversation room
- `send_message` - Send message to conversation
- `typing_start` - Start typing indicator
- `typing_stop` - Stop typing indicator

### Frontend Components

#### 1. Chat Module (`chatModule.vue`)
Main chat interface with conversation list and message area.

#### 2. Chat Sidebar (`ChatSidebar.vue`)
Navigation component for quick access to conversations.

#### 3. Chat API (`chatApi.js`)
Service layer for HTTP API communication.

#### 4. Chat Socket (`useChatSocket.js`)
Composable for WebSocket functionality and real-time features.

#### 5. Chat Store (`chatStore.js`)
Pinia store for state management.

## Installation & Setup

### Backend Setup

1. **Install Dependencies**
```bash
cd API-Hybrid-Auth
npm install socket.io
```

2. **Create Database Tables**
```bash
mysql -u your_user -p your_database < scripts/create-chat-tables.sql
```

3. **Environment Variables**
Ensure your `.env` file contains:
```
JWT_SECRET=your_jwt_secret_here
PORT=3001
```

4. **Start Server**
```bash
npm start
```

### Frontend Setup

1. **Install Dependencies**
```bash
cd Hybrid-ERP
npm install socket.io-client
```

2. **Add Chat Route**
Update your main router configuration to include the chat module.

3. **Configure API Base URL**
Set `VUE_APP_API_URL` in your environment variables.

## Usage

### Creating a Conversation

1. Click the "+" button in the chat header
2. Enter conversation title
3. Select participants from the user list
4. Click "Create"

### Sending Messages

1. Select a conversation from the sidebar
2. Type your message in the input field
3. Press Enter or click the send button

### Real-time Features

- Messages appear instantly for all participants
- Typing indicators show when users are composing
- Unread message counts update in real-time
- Conversation list updates automatically

## Security Features

### Authentication
- JWT token validation for all API requests
- WebSocket authentication before allowing chat access
- User permission checks for conversation access

### Data Validation
- Input sanitization for messages
- Participant verification for conversation access
- Rate limiting for message sending

## Performance Considerations

### Database Optimization
- Indexed queries for fast message retrieval
- Pagination for large message histories
- Efficient conversation listing queries

### WebSocket Management
- Connection pooling and reuse
- Automatic reconnection handling
- Memory-efficient event handling

## Troubleshooting

### Common Issues

1. **WebSocket Connection Failed**
   - Check if the backend server is running
   - Verify CORS configuration
   - Check network connectivity

2. **Messages Not Sending**
   - Verify JWT token is valid
   - Check user permissions
   - Ensure conversation exists

3. **Real-time Updates Not Working**
   - Check WebSocket connection status
   - Verify event handlers are properly registered
   - Check browser console for errors

### Debug Mode
Enable debug logging by setting:
```javascript
localStorage.setItem('chat_debug', 'true')
```

## Future Enhancements

### Planned Features
- File and image sharing
- Message reactions and emojis
- Voice and video calls
- Message encryption
- Advanced search and filtering
- Conversation archiving
- Bot integration for automated responses

### Scalability Improvements
- Redis for session management
- Message queuing for high-traffic scenarios
- Horizontal scaling support
- Load balancing for WebSocket connections

## Support

For technical support or questions about the chat system implementation, please refer to the development team or create an issue in the project repository.
