require('dotenv').config({ path: require('path').resolve(__dirname, '../../.env') });
const express = require('express');
const cors = require('cors');
const http = require('http');
const socketIo = require('socket.io');
const logsMiddleware = require('./middleware/logsMiddleware');
const authRoutes = require('./routes/authRoutes');
const userRoutes = require('./routes/userRoutes');
const superAdminRoutes = require('./routes/superAdminRoutes');
const chatRoutes = require('./routes/chatRoutes');
// Newly mounted route groups
const roleRoutes = require('./routes/roleRoutes');
const permissionRoutes = require('./routes/permissionRoutes');
const userPermissionRoutes = require('./routes/userPermissionRoutes');
const rolePermissionRoutes = require('./routes/rolePermissionRoutes');
const superAdminUserRoutes = require('./routes/superAdminUserRoutes');
const notificationRoutes = require('./routes/notificationRoutes');
const approvalRoutes = require('./routes/approvalRoutes');
const crudOperationsRoutes = require('./routes/crudOperationsRoutes');
const workflowRoutes = require('./routes/workflowRoutes');
const storeRoutes = require('./routes/storeRoutes');
const authenticateToken = require('./middleware/authMiddleware');
const { authenticateSuperAdmin } = require('./middleware/superAdminMiddleware');
// Check if chatRoutes exports both router and fileRouter
let chatRoutesRouter = chatRoutes;
let fileRouter = null;
try {
    const chatRoutesModule = require('./routes/chatRoutes');
    if (chatRoutesModule.router && chatRoutesModule.fileRouter) {
        chatRoutesRouter = chatRoutesModule.router;
        fileRouter = chatRoutesModule.fileRouter;
    }
} catch (e) {
    // Use default chatRoutes if destructuring fails
}
const db = require('./config/db');
const setupWebSocket = require('./websocket');
const errorHandler = require('./middleware/errorHandler');
const createApprovalTables = require('./scripts/createApprovalTables');

const app = express();
const server = http.createServer(app);
const io = socketIo(server, {
    cors: {
        origin: "*",
        methods: ["GET", "POST", "PUT", "DELETE", "PATCH", "OPTIONS"]
    }
});

// Explicit CORS configuration
const corsOptions = {
    origin: [
        'http://localhost:9000',  // Frontend URL
        'http://localhost:9001',  // Frontend URL (alternative port)
        'http://localhost:3001',  // Auth service URL
        'http://127.0.0.1:9000',
        'http://127.0.0.1:9001',  // Frontend URL (alternative port)
        'http://127.0.0.1:3001'
    ],
    methods: ['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS'],
    allowedHeaders: ['Content-Type', 'Authorization', 'x-user-id', 'x-user-name', 'x-username', 'x-role', 'x-module-name', 'x-service-token', 'x-correlation-id'],
    credentials: true
};

app.use(cors(corsOptions));
app.use(express.json());
app.use(logsMiddleware); // Logs every API request
app.use(errorHandler);

// Routes
app.use('/api/auth', authRoutes);
app.use('/api/users', userRoutes); // User routes - some endpoints require auth, some don't (see userRoutes.js)
app.use('/api/superadmin', superAdminRoutes);
app.use('/api/superadmin/users', superAdminUserRoutes); // Superadmin-managed users
app.use('/api/roles', roleRoutes);
app.use('/api/permissions', permissionRoutes);
app.use('/api/user-permissions', userPermissionRoutes);
app.use('/api/role-permissions', rolePermissionRoutes);
app.use('/api/chat', authenticateToken, chatRoutesRouter); // Secure chat routes
if (fileRouter) {
    app.use('/files', fileRouter); // Mount file router for public file access
}
app.use('/api/notifications', notificationRoutes);
app.use('/api/approvals', approvalRoutes); // Legacy approval system
app.use('/api/crud-operations', crudOperationsRoutes);
app.use('/api/workflow', workflowRoutes); // Central workflow management
app.use('/api/stores', storeRoutes); // Stores/Branches endpoint

app.get('/', (req, res) => {
    res.send('Auth API is running');
});

// Catch-all route for debugging
app.use((req, res) => {
    console.log('Unhandled route:', {
        method: req.method,
        path: req.path,
        headers: req.headers,
        body: req.body
    });
    res.status(404).json({ 
        message: 'Route not found', 
        details: {
            method: req.method,
            path: req.path
        }
    });
});

// Initialize WebSocket
setupWebSocket(io);

const PORT = process.env.AUTH_API_PORT || 3001;

const startServer = async () => {
    try {
        // Skip database setup for now - will be handled later
        console.log('⏭️  Skipping database table creation during startup');
        console.log('📝 To set up database tables, run: npm run setup-db');
        // Ensure notifications table exists
        try {
            const ensureNotificationsTable = require('./scripts/createNotificationTable');
            if (typeof ensureNotificationsTable === 'function') {
                await ensureNotificationsTable();
            }
        } catch (e) {
            console.warn('Notification table ensure failed (non-blocking):', e.message);
        }

        // Start the server with error handling
        server.on('error', (error) => {
            if (error.code === 'EADDRINUSE') {
                console.error(`Port ${PORT} is already in use. Please make sure no other services are running on this port.`);
                process.exit(1);
            } else {
                console.error('Server error:', error);
                process.exit(1);
            }
        });

        server.listen(PORT, () => {
            console.log(`Server running on port ${PORT}`);

            // Start the execution engine
            try {
                if (process.env.DISABLE_EXECUTION_ENGINE === 'true') {
                    console.warn('ExecutionEngine is disabled via DISABLE_EXECUTION_ENGINE=true');
                } else {
                    // Verify DB connectivity before starting the engine
                    const startEngine = async () => {
                        try {
                            await db.query('SELECT 1');
                const ExecutionEngine = require('./services/executionEngine');
                ExecutionEngine.start();
                            console.log('✅ ExecutionEngine started after DB check');
                            return true;
                        } catch (e) {
                            console.warn('⏳ DB not ready, ExecutionEngine start deferred:', e.message);
                            return false;
                        }
                    };
                    // Try immediately, then retry every 30s until success
                    startEngine().then((started) => {
                        if (!started) {
                            const interval = setInterval(async () => {
                                const ok = await startEngine();
                                if (ok) clearInterval(interval);
                            }, 30000);
                        }
                    });
                }
            } catch (error) {
                console.warn('ExecutionEngine failed to start (non-blocking):', error.message);
            }
        });
    } catch (error) {
        console.error('Failed to start server:', error);
        process.exit(1);
    }
};

startServer();

// Graceful shutdown
process.on('SIGTERM', () => {
    const ExecutionEngine = require('./services/executionEngine');
    ExecutionEngine.stop();
    process.exit(0);
});