// CRUD Operations Configuration
// This file defines all the CRUD operations that can be performed in the system
// and whether they require approval or not

const crudOperationsConfig = {
    // Supply Chain Module Operations
    supply_chain: {
        // Purchase Order Operations
        create_purchase_order: {
            entity_type: 'purchase_order',
            requires_approval: true,
            approval_workflow_id: 1,
            description: 'Create new purchase order - requires approval for amounts over $10,000',
            conditional_approval: {
                field: 'total_amount',
                threshold: 10000,
                operator: '>',
                message: 'Purchase orders over $10,000 require approval'
            }
        },
        update_purchase_order: {
            entity_type: 'purchase_order',
            requires_approval: true,
            approval_workflow_id: 1,
            description: 'Update existing purchase order - requires approval for significant changes'
        },
        delete_purchase_order: {
            entity_type: 'purchase_order',
            requires_approval: true,
            approval_workflow_id: 1,
            description: 'Delete purchase order - requires approval for audit trail'
        },
        approve_purchase_order: {
            entity_type: 'purchase_order',
            requires_approval: false,
            description: 'Approve purchase order - no additional approval required'
        },

        // Supplier Operations
        create_supplier: {
            entity_type: 'supplier',
            requires_approval: true,
            approval_workflow_id: 2,
            description: 'Create new supplier - requires approval for vendor management'
        },
        update_supplier: {
            entity_type: 'supplier',
            requires_approval: true,
            approval_workflow_id: 2,
            description: 'Update supplier information - requires approval for data integrity'
        },
        delete_supplier: {
            entity_type: 'supplier',
            requires_approval: true,
            approval_workflow_id: 2,
            description: 'Delete supplier - requires approval for audit trail'
        },

        // Raw Material Operations
        create_raw_material: {
            entity_type: 'raw_material',
            requires_approval: false,
            description: 'Create raw material - no approval required'
        },
        update_raw_material: {
            entity_type: 'raw_material',
            requires_approval: false,
            description: 'Update raw material - no approval required'
        },
        delete_raw_material: {
            entity_type: 'raw_material',
            requires_approval: true,
            approval_workflow_id: 1,
            description: 'Delete raw material - requires approval for inventory control'
        },

        // Vehicle Operations
        create_vehicle: {
            entity_type: 'vehicle',
            requires_approval: false,
            description: 'Create vehicle record - no approval required'
        },
        update_vehicle: {
            entity_type: 'vehicle',
            requires_approval: false,
            description: 'Update vehicle information - no approval required'
        },
        delete_vehicle: {
            entity_type: 'vehicle',
            requires_approval: true,
            approval_workflow_id: 1,
            description: 'Delete vehicle - requires approval for fleet management'
        },

        // Weighbridge Operations
        create_weighbridge_record: {
            entity_type: 'weighbridge_record',
            requires_approval: false,
            description: 'Create weighbridge record - no approval required'
        },
        update_weighbridge_record: {
            entity_type: 'weighbridge_record',
            requires_approval: true,
            approval_workflow_id: 1,
            description: 'Update weighbridge record - requires approval for data integrity'
        },

        // Quality Inspection Operations
        create_quality_inspection: {
            entity_type: 'quality_inspection',
            requires_approval: false,
            description: 'Create quality inspection - no approval required'
        },
        update_quality_inspection: {
            entity_type: 'quality_inspection',
            requires_approval: true,
            approval_workflow_id: 1,
            description: 'Update quality inspection - requires approval for data integrity'
        }
    },

    // Finance Module Operations
    finance: {
        create_invoice: {
            entity_type: 'invoice',
            requires_approval: true,
            approval_workflow_id: 3,
            description: 'Create invoice - requires approval for billing control'
        },
        update_invoice: {
            entity_type: 'invoice',
            requires_approval: true,
            approval_workflow_id: 3,
            description: 'Update invoice - requires approval for billing integrity'
        },
        delete_invoice: {
            entity_type: 'invoice',
            requires_approval: true,
            approval_workflow_id: 3,
            description: 'Delete invoice - requires approval for audit trail'
        },
        create_payment: {
            entity_type: 'payment',
            requires_approval: true,
            approval_workflow_id: 3,
            description: 'Create payment record - requires approval for financial control'
        },
        update_payment: {
            entity_type: 'payment',
            requires_approval: true,
            approval_workflow_id: 3,
            description: 'Update payment record - requires approval for financial integrity'
        }
    },

    // Operations Module Operations
    operations: {
        create_bom: {
            entity_type: 'bill_of_materials',
            requires_approval: true,
            approval_workflow_id: 4,
            description: 'Create bill of materials - requires approval for production control'
        },
        update_bom: {
            entity_type: 'bill_of_materials',
            requires_approval: true,
            approval_workflow_id: 4,
            description: 'Update bill of materials - requires approval for production integrity'
        },
        delete_bom: {
            entity_type: 'bill_of_materials',
            requires_approval: true,
            approval_workflow_id: 4,
            description: 'Delete bill of materials - requires approval for production audit'
        },
        create_work_order: {
            entity_type: 'work_order',
            requires_approval: false,
            description: 'Create work order - no approval required'
        },
        update_work_order: {
            entity_type: 'work_order',
            requires_approval: false,
            description: 'Update work order - no approval required'
        }
    },

    // Accounts Module Operations
    accounts: {
        create_chart_of_accounts: {
            entity_type: 'chart_of_accounts',
            requires_approval: true,
            approval_workflow_id: 5,
            description: 'Create chart of accounts - requires approval for financial structure'
        },
        update_chart_of_accounts: {
            entity_type: 'chart_of_accounts',
            requires_approval: true,
            approval_workflow_id: 5,
            description: 'Update chart of accounts - requires approval for financial integrity'
        },
        create_journal_entry: {
            entity_type: 'journal_entry',
            requires_approval: true,
            approval_workflow_id: 5,
            description: 'Create journal entry - requires approval for accounting control'
        },
        update_journal_entry: {
            entity_type: 'journal_entry',
            requires_approval: true,
            approval_workflow_id: 5,
            description: 'Update journal entry - requires approval for accounting integrity'
        }
    }
};

// Helper function to get all operations as a flat array
const getAllOperations = () => {
    const operations = [];

    for (const [module, moduleOperations] of Object.entries(crudOperationsConfig)) {
        for (const [operation, config] of Object.entries(moduleOperations)) {
            operations.push({
                module,
                operation,
                entity_type: config.entity_type,
                requires_approval: config.requires_approval,
                approval_workflow_id: config.approval_workflow_id,
                description: config.description,
                conditional_approval: config.conditional_approval
            });
        }
    }

    return operations;
};

// Helper function to get operations by module
const getOperationsByModule = (module) => {
    return crudOperationsConfig[module] || {};
};

// Helper function to check if operation requires approval
const checkApprovalRequired = (module, operation, data = null) => {
    const moduleOperations = crudOperationsConfig[module];
    if (!moduleOperations) return { requires_approval: false };

    const operationConfig = moduleOperations[operation];
    if (!operationConfig) return { requires_approval: false };

    let requiresApproval = operationConfig.requires_approval;

    // Check conditional approval logic
    if (data && operationConfig.conditional_approval) {
        const { field, threshold, operator } = operationConfig.conditional_approval;
        const value = data[field];

        if (value !== undefined) {
            switch (operator) {
                case '>':
                    requiresApproval = value > threshold;
                    break;
                case '>=':
                    requiresApproval = value >= threshold;
                    break;
                case '<':
                    requiresApproval = value < threshold;
                    break;
                case '<=':
                    requiresApproval = value <= threshold;
                    break;
                case '==':
                    requiresApproval = value == threshold;
                    break;
                case '!=':
                    requiresApproval = value != threshold;
                    break;
            }
        }
    }

    return {
        requires_approval: requiresApproval,
        workflow_id: operationConfig.approval_workflow_id,
        description: operationConfig.description
    };
};

module.exports = {
    crudOperationsConfig,
    getAllOperations,
    getOperationsByModule,
    checkApprovalRequired
};