const db = require('../config/db');

// Helper function to generate request key
const makeRequestKey = () => {
    return 'AR-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9);
};

// Get all approval requests
exports.getAllRequests = async(req, res) => {
    try {
        const [requests] = await db.query(`
            SELECT 
                ar.*,
                aw.name as workflow_name,
                aw.module as workflow_module,
                aw.operation as workflow_operation
            FROM approval_requests ar
            LEFT JOIN approval_workflows aw ON ar.workflow_id = aw.id
            ORDER BY ar.created_at DESC
        `);

        res.json({
            success: true,
            data: requests
        });
    } catch (error) {
        console.error('Error fetching approval requests:', error);
        res.status(500).json({ error: 'Failed to fetch approval requests' });
    }
};

// Get approval request by ID with details
exports.getRequestById = async(req, res) => {
    const { id } = req.params;

    try {
        // Get request details
        const [requests] = await db.query(`
            SELECT 
                ar.*,
                aw.name as workflow_name,
                aw.module as workflow_module,
                aw.operation as workflow_operation
            FROM approval_requests ar
            LEFT JOIN approval_workflows aw ON ar.workflow_id = aw.id
            WHERE ar.id = ?
        `, [id]);

        if (requests.length === 0) {
            return res.status(404).json({ error: 'Approval request not found' });
        }

        const request = requests[0];

        // Get approval actions
        const [actions] = await db.query(`
            SELECT 
                aa.*,
                al.name as level_name,
                al.order as level_order
            FROM approval_actions aa
            LEFT JOIN approval_levels al ON aa.level_id = al.id
            WHERE aa.request_id = ?
            ORDER BY aa.created_at ASC
        `, [id]);

        // Get attachments
        const [attachments] = await db.query(
            'SELECT * FROM approval_attachments WHERE request_id = ? ORDER BY created_at ASC', [id]
        );

        request.actions = actions;
        request.attachments = attachments;

        res.json({
            success: true,
            data: request
        });
    } catch (error) {
        console.error('Error fetching approval request:', error);
        res.status(500).json({ error: 'Failed to fetch approval request' });
    }
};

// Create new approval request
exports.createRequest = async(req, res) => {
    const {
        workflow_id,
        title,
        description,
        module,
        operation,
        entity_type,
        entity_id,
        requester_id,
        requester_name,
        requester_email,
        requester_department,
        priority,
        attachments
    } = req.body;

    // Validate required fields
    if (!workflow_id || !title || !module || !operation) {
        return res.status(400).json({
            error: 'Missing required fields',
            required: ['workflow_id', 'title', 'module', 'operation']
        });
    }

    try {
        // Check if workflow exists and is active
        const [workflows] = await db.query(
            'SELECT * FROM approval_workflows WHERE id = ? AND status = "active"', [workflow_id]
        );

        if (workflows.length === 0) {
            return res.status(400).json({ error: 'Invalid or inactive workflow' });
        }

        const workflow = workflows[0];

        // Get approval levels
        const [levels] = await db.query(
            'SELECT * FROM approval_levels WHERE workflow_id = ? ORDER BY `order`', [workflow_id]
        );

        if (levels.length === 0) {
            return res.status(400).json({ error: 'Workflow has no approval levels configured' });
        }

        // Create approval request
        const [requestResult] = await db.query(
            `INSERT INTO approval_requests 
                (workflow_id, title, description, module, operation, entity_type, entity_id,
                 requester_id, requester_name, requester_email, requester_department, priority, status)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`, [
                workflow_id,
                title,
                description,
                module,
                operation,
                entity_type || null,
                entity_id || null,
                requester_id || req.headers['user-id'] || null,
                requester_name,
                requester_email,
                requester_department,
                priority || 'medium',
                'pending'
            ]
        );

        const requestId = requestResult.insertId;

        // Create attachments if provided
        if (attachments && Array.isArray(attachments)) {
            for (const attachment of attachments) {
                await db.query(
                    `INSERT INTO approval_attachments 
                        (request_id, filename, file_path, file_size, mime_type)
                        VALUES (?, ?, ?, ?, ?)`, [
                        requestId,
                        attachment.filename,
                        attachment.file_path,
                        attachment.file_size,
                        attachment.mime_type
                    ]
                );
            }
        }

        // Create notifications for first level approvers
        const firstLevel = levels[0];
        if (firstLevel.roles && firstLevel.roles.length > 0) {
            await db.query(
                `INSERT INTO approval_notifications 
                    (request_id, user_id, notification_type, title, message, status)
                    VALUES (?, ?, ?, ?, ?, ?)`, [
                    requestId,
                    null, // Will be set based on roles
                    'approval_required',
                    'New Approval Request',
                    `${title} requires your approval`
                ]
            );
        }

        // Get the created request
        const [requests] = await db.query(
            'SELECT * FROM approval_requests WHERE id = ?', [requestId]
        );

        res.status(201).json({
            success: true,
            message: 'Approval request created successfully',
            data: requests[0]
        });
    } catch (error) {
        console.error('Error creating approval request:', error);
        res.status(500).json({ error: 'Failed to create approval request' });
    }
};

// Approve/Reject request
exports.approveRequest = async(req, res) => {
    const { id } = req.params;
    const { status, comments, level_id } = req.body;

    if (!status || !level_id) {
        return res.status(400).json({
            error: 'Missing required fields',
            required: ['status', 'level_id']
        });
    }

    try {
        // Get request details
        const [requests] = await db.query(
            'SELECT * FROM approval_requests WHERE id = ?', [id]
        );

        if (requests.length === 0) {
            return res.status(404).json({ error: 'Approval request not found' });
        }

        const request = requests[0];

        // Get approval level details
        const [levels] = await db.query(
            'SELECT * FROM approval_levels WHERE id = ? AND workflow_id = ?', [level_id, request.workflow_id]
        );

        if (levels.length === 0) {
            return res.status(400).json({ error: 'Invalid approval level' });
        }

        // Create approval action
        await db.query(
            `INSERT INTO approval_actions 
                (request_id, level_id, action, comments, approver_id, approver_name)
                VALUES (?, ?, ?, ?, ?, ?)`, [
                id,
                level_id,
                status,
                comments || '',
                req.headers['user-id'] || null,
                req.headers['user-name'] || 'Unknown'
            ]
        );

        // Update request status based on approval result
        if (status === 'approved') {
            // Check if there are more levels
            const [nextLevels] = await db.query(
                'SELECT * FROM approval_levels WHERE workflow_id = ? AND `order` = ?', [request.workflow_id, levels[0].order + 1]
            );

            if (nextLevels.length > 0) {
                // Move to next level
                await db.query(
                    'UPDATE approval_requests SET current_level = ?, status = ? WHERE id = ?', [nextLevels[0].order, 'under_review', id]
                );

                // Create notification for next level
                await db.query(
                    `INSERT INTO approval_notifications 
                        (request_id, user_id, notification_type, title, message, status)
                        VALUES (?, ?, ?, ?, ?, ?)`, [
                        id,
                        null,
                        'approval_required',
                        'Approval Required',
                        `${request.title} requires your approval`
                    ]
                );
            } else {
                // Final approval
                await db.query(
                    'UPDATE approval_requests SET status = ? WHERE id = ?', ['approved', id]
                );

                // Trigger execution of queued operations
                await triggerOperationExecution(id);

                // Notify requester
                await notifyRequester(id, 'approved');
            }
        } else if (status === 'rejected') {
            // Reject the request
            await db.query(
                'UPDATE approval_requests SET status = ? WHERE id = ?', ['rejected', id]
            );
        }

        // Get updated request
        const [updatedRequests] = await db.query(
            'SELECT * FROM approval_requests WHERE id = ?', [id]
        );

        res.json({
            success: true,
            message: `Request ${status} successfully`,
            data: updatedRequests[0]
        });
    } catch (error) {
        console.error('Error processing approval:', error);
        res.status(500).json({ error: 'Failed to process approval' });
    }
};

// Get pending approvals for a user
exports.getPendingApprovals = async(req, res) => {
    const { user_id, module, priority } = req.query;

    try {
        let query = `
            SELECT 
                ar.*,
                aw.name as workflow_name,
                aw.module as workflow_module,
                aw.operation as workflow_operation,
                al.name as level_name,
                al.order as level_order
            FROM approval_requests ar
            LEFT JOIN approval_workflows aw ON ar.workflow_id = aw.id
            LEFT JOIN approval_levels al ON ar.workflow_id = al.workflow_id AND al.order = ar.current_level
            WHERE ar.status IN ('pending', 'under_review')
        `;

        const params = [];

        if (user_id) {
            query += ' AND ar.requester_id = ?';
            params.push(user_id);
        }

        if (module) {
            query += ' AND ar.module = ?';
            params.push(module);
        }

        if (priority) {
            query += ' AND ar.priority = ?';
            params.push(priority);
        }

        query += ' ORDER BY ar.priority DESC, ar.created_at ASC';

        const [requests] = await db.query(query, params);

        res.json({
            success: true,
            data: requests
        });
    } catch (error) {
        console.error('Error fetching pending approvals:', error);
        res.status(500).json({ error: 'Failed to fetch pending approvals' });
    }
};

// Get approval statistics
exports.getApprovalStats = async(req, res) => {
    try {
        const [stats] = await db.query(`
            SELECT 
                COUNT(*) as total_requests,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_requests,
                SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_requests,
                SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected_requests,
                SUM(CASE WHEN status = 'under_review' THEN 1 ELSE 0 END) as under_review_requests
            FROM approval_requests
        `);

        res.json({
            success: true,
            data: stats[0]
        });
    } catch (error) {
        console.error('Error fetching approval stats:', error);
        res.status(500).json({ error: 'Failed to fetch approval statistics' });
    }
};

// Cancel approval request
exports.cancelRequest = async(req, res) => {
    const { id } = req.params;
    const userId = req.headers['user-id'];

    try {
        // Check if request exists and belongs to user
        const [requests] = await db.query(
            'SELECT * FROM approval_requests WHERE id = ? AND requester_id = ?', [id, userId]
        );

        if (requests.length === 0) {
            return res.status(404).json({ error: 'Approval request not found or not authorized' });
        }

        const request = requests[0];

        // Check if request can be cancelled
        if (request.status !== 'pending' && request.status !== 'under_review') {
            return res.status(400).json({ error: 'Request cannot be cancelled in current status' });
        }

        // Update request status
        await db.query(
            'UPDATE approval_requests SET status = ? WHERE id = ?', ['cancelled', id]
        );

        res.json({
            success: true,
            message: 'Approval request cancelled successfully'
        });
    } catch (error) {
        console.error('Error cancelling approval request:', error);
        res.status(500).json({ error: 'Failed to cancel approval request' });
    }
};

// CRUD Operations Registry Functions

// Get operations registry
exports.getOperationsRegistry = async(req, res) => {
    try {
        const [operations] = await db.query(`
            SELECT * FROM crud_operations_registry 
            ORDER BY module, operation, created_at DESC
        `);

        res.json({
            success: true,
            data: operations
        });
    } catch (error) {
        console.error('Error fetching operations registry:', error);
        res.status(500).json({ error: 'Failed to fetch operations registry' });
    }
};

// Register new operation
exports.registerOperation = async(req, res) => {
    const {
        module,
        operation,
        entity_type,
        requires_approval,
        approval_workflow_id,
        description
    } = req.body;

    if (!module || !operation || !entity_type) {
        return res.status(400).json({
            error: 'Missing required fields',
            required: ['module', 'operation', 'entity_type']
        });
    }

    try {
        const [result] = await db.query(
            `INSERT INTO crud_operations_registry 
                (module, operation, entity_type, requires_approval, approval_workflow_id, description, created_by)
                VALUES (?, ?, ?, ?, ?, ?, ?)`, [
                module,
                operation,
                entity_type,
                requires_approval || false,
                approval_workflow_id || null,
                description || null,
                req.headers['user-id'] || null
            ]
        );

        res.status(201).json({
            success: true,
            message: 'Operation registered successfully',
            data: { id: result.insertId }
        });
    } catch (error) {
        console.error('Error registering operation:', error);
        res.status(500).json({ error: 'Failed to register operation' });
    }
};

// Update operation
exports.updateOperation = async(req, res) => {
    const { id } = req.params;
    const {
        module,
        operation,
        entity_type,
        requires_approval,
        approval_workflow_id,
        description
    } = req.body;

    try {
        await db.query(
            `UPDATE crud_operations_registry 
                SET module = ?, operation = ?, entity_type = ?, requires_approval = ?, 
                    approval_workflow_id = ?, description = ?
                WHERE id = ?`, [
                module,
                operation,
                entity_type,
                requires_approval,
                approval_workflow_id,
                description,
                id
            ]
        );

        res.json({
            success: true,
            message: 'Operation updated successfully'
        });
    } catch (error) {
        console.error('Error updating operation:', error);
        res.status(500).json({ error: 'Failed to update operation' });
    }
};

// Delete operation
exports.deleteOperation = async(req, res) => {
    const { id } = req.params;

    try {
        await db.query('DELETE FROM crud_operations_registry WHERE id = ?', [id]);

        res.json({
            success: true,
            message: 'Operation deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting operation:', error);
        res.status(500).json({ error: 'Failed to delete operation' });
    }
};

// Helper function to trigger operation execution
async function triggerOperationExecution(requestId) {
    try {
        const ExecutionEngine = require('../services/executionEngine');

        // Start the execution engine if not running
        if (!ExecutionEngine.isRunning) {
            ExecutionEngine.start();
        }

        console.log(`🚀 Triggered execution for approval request ${requestId}`);
    } catch (error) {
        console.error('Error triggering operation execution:', error);
    }
}

// Helper function to notify requester
async function notifyRequester(requestId, status) {
    try {
        const [requests] = await db.query(`
            SELECT * FROM approval_requests WHERE id = ?
        `, [requestId]);

        if (requests.length > 0) {
            const request = requests[0];

            // Create notification
            await db.query(`
                INSERT INTO approval_notifications 
                (request_id, user_id, notification_type, title, message, status)
                VALUES (?, ?, ?, ?, ?, ?)
            `, [
                requestId,
                request.requester_id,
                'approval_complete',
                `Request ${status.charAt(0).toUpperCase() + status.slice(1)}`,
                `Your request "${request.title}" has been ${status}.`,
                'unread'
            ]);

            console.log(`📧 Notification sent to requester ${request.requester_id}`);
        }
    } catch (error) {
        console.error('Error notifying requester:', error);
    }
}

// Get all queued operations
exports.getQueuedOperations = async(req, res) => {
    try {
        const OperationQueueService = require('../services/operationQueueService');
        const operations = await OperationQueueService.getPendingOperations();

        res.json({
            success: true,
            data: operations
        });
    } catch (error) {
        console.error('Error fetching queued operations:', error);
        res.status(500).json({ error: 'Failed to fetch queued operations' });
    }
};

// Queue a new operation
exports.queueOperation = async(req, res) => {
    try {
        const OperationQueueService = require('../services/operationQueueService');
        const operationId = await OperationQueueService.queueOperation(req.body);

        res.status(201).json({
            success: true,
            message: 'Operation queued successfully',
            data: { id: operationId }
        });
    } catch (error) {
        console.error('Error queuing operation:', error);
        res.status(500).json({ error: 'Failed to queue operation' });
    }
};

// Get queued operation by ID
exports.getQueuedOperationById = async(req, res) => {
    try {
        const { id } = req.params;
        const OperationQueueService = require('../services/operationQueueService');
        const operation = await OperationQueueService.getOperationById(id);

        if (!operation) {
            return res.status(404).json({ error: 'Operation not found' });
        }

        res.json({
            success: true,
            data: operation
        });
    } catch (error) {
        console.error('Error fetching queued operation:', error);
        res.status(500).json({ error: 'Failed to fetch queued operation' });
    }
};

// Force execute a queued operation
exports.forceExecuteOperation = async(req, res) => {
    try {
        const { id } = req.params;
        const ExecutionEngine = require('../services/executionEngine');

        await ExecutionEngine.forceExecuteOperation(parseInt(id));

        res.json({
            success: true,
            message: 'Operation execution triggered successfully'
        });
    } catch (error) {
        console.error('Error force executing operation:', error);
        res.status(500).json({ error: 'Failed to execute operation' });
    }
};

// Check if approval is required for an operation
exports.checkApprovalRequired = async(req, res) => {
    try {
        const { module, operation, data } = req.query;

        if (!module || !operation) {
            return res.status(400).json({
                error: 'Missing required fields',
                required: ['module', 'operation']
            });
        }

        // Get operation from registry
        const [operations] = await db.query(`
            SELECT * FROM crud_operations_registry 
            WHERE module = ? AND operation = ?
        `, [module, operation]);

        if (operations.length === 0) {
            // Operation not found in registry, default to no approval required
            return res.json({
                requires_approval: false,
                message: 'Operation not found in registry'
            });
        }

        const operationConfig = operations[0];

        // Check if approval is required
        if (!operationConfig.requires_approval) {
            return res.json({
                requires_approval: false,
                message: 'Operation does not require approval'
            });
        }

        // If data is provided, you can add conditional logic here
        // For example, check amount thresholds, user roles, etc.
        let requiresApproval = operationConfig.requires_approval;

        if (data) {
            try {
                const parsedData = JSON.parse(data);

                // Example: Check if purchase order amount exceeds threshold
                if (module === 'supply_chain' && operation === 'create_purchase_order') {
                    const totalAmount = parsedData.total_amount || 0;
                    if (totalAmount > 10000) { // $10,000 threshold
                        requiresApproval = true;
                    } else {
                        requiresApproval = false;
                    }
                }
            } catch (error) {
                console.error('Error parsing data:', error);
            }
        }

        res.json({
            requires_approval: requiresApproval,
            workflow_id: operationConfig.approval_workflow_id,
            message: requiresApproval ? 'Approval required' : 'No approval required'
        });

    } catch (error) {
        console.error('Error checking approval requirements:', error);
        res.status(500).json({ error: 'Failed to check approval requirements' });
    }
};