const db = require('../config/db');

// Helper function to generate workflow key
const makeWorkflowKey = () => {
    return 'WF-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9);
};

// Get all approval workflows
exports.getAllWorkflows = async(req, res) => {
    try {
        console.log('Fetching workflows...');
        const [workflows] = await db.query(`
            SELECT 
                w.*,
                COUNT(al.id) as level_count
            FROM approval_workflows w
            LEFT JOIN approval_levels al ON w.id = al.workflow_id
            GROUP BY w.id
            ORDER BY w.created_at DESC
        `);

        // Get approval levels for each workflow
        for (let workflow of workflows) {
            const [levels] = await db.query(
                'SELECT * FROM approval_levels WHERE workflow_id = ? ORDER BY `order`', [workflow.id]
            );

            // Parse JSON roles for each level
            levels.forEach(level => {
                if (level.roles) {
                    try {
                        level.roles = JSON.parse(level.roles);
                    } catch (e) {
                        level.roles = [];
                    }
                }
            });

            workflow.approval_levels = levels;
        }

        console.log('Workflows found:', workflows.length);
        res.json({
            success: true,
            data: workflows
        });
    } catch (error) {
        console.error('Error fetching workflows:', error);
        res.status(500).json({ error: 'Failed to fetch workflows', details: error.message });
    }
};

// Get workflow by ID with levels
exports.getWorkflowById = async(req, res) => {
    const { id } = req.params;

    try {
        // Get workflow details
        const [workflows] = await db.query(
            'SELECT * FROM approval_workflows WHERE id = ?', [id]
        );

        if (workflows.length === 0) {
            return res.status(404).json({ error: 'Workflow not found' });
        }

        const workflow = workflows[0];

        // Get approval levels
        const [levels] = await db.query(
            'SELECT * FROM approval_levels WHERE workflow_id = ? ORDER BY `order`', [id]
        );

        // Parse JSON roles for each level
        levels.forEach(level => {
            if (level.roles) {
                level.roles = JSON.parse(level.roles);
            }
        });

        workflow.approval_levels = levels;

        res.json({
            success: true,
            data: workflow
        });
    } catch (error) {
        console.error('Error fetching workflow:', error);
        res.status(500).json({ error: 'Failed to fetch workflow' });
    }
};

// Create new workflow
exports.createWorkflow = async(req, res) => {
    const {
        name,
        module,
        operation,
        description,
        auto_approve,
        sequential_approval,
        approval_levels
    } = req.body;

    // Validate required fields
    if (!name || !module || !operation) {
        return res.status(400).json({
            error: 'Missing required fields',
            required: ['name', 'module', 'operation']
        });
    }

    try {
        // Create workflow
        const [workflowResult] = await db.query(
            `INSERT INTO approval_workflows 
                (name, module, operation, description, auto_approve, sequential_approval, created_by)
                VALUES (?, ?, ?, ?, ?, ?, ?)`, [
                name,
                module,
                operation,
                description || null,
                auto_approve || false,
                sequential_approval || false,
                req.headers['user-id'] || null
            ]
        );

        const workflowId = workflowResult.insertId;

        // Create approval levels if provided
        if (approval_levels && Array.isArray(approval_levels) && approval_levels.length > 0) {
            for (const level of approval_levels) {
                await db.query(
                    `INSERT INTO approval_levels 
                        (workflow_id, name, \`order\`, required, roles)
                        VALUES (?, ?, ?, ?, ?)`, [
                        workflowId,
                        level.name,
                        level.order,
                        level.required !== undefined ? level.required : true,
                        JSON.stringify(level.roles || [])
                    ]
                );
            }
        }

        // Get the created workflow with levels
        const [workflows] = await db.query(
            'SELECT * FROM approval_workflows WHERE id = ?', [workflowId]
        );

        const [levels] = await db.query(
            'SELECT * FROM approval_levels WHERE workflow_id = ? ORDER BY `order`', [workflowId]
        );

        // Parse JSON roles
        levels.forEach(level => {
            if (level.roles) {
                level.roles = JSON.parse(level.roles);
            }
        });

        const workflow = {...workflows[0], approval_levels: levels };

        res.status(201).json({
            success: true,
            message: 'Workflow created successfully',
            data: workflow
        });
    } catch (error) {
        console.error('Error creating workflow:', error);
        res.status(500).json({ error: 'Failed to create workflow' });
    }
};

// Update workflow
exports.updateWorkflow = async(req, res) => {
    const { id } = req.params;
    const {
        name,
        module,
        operation,
        description,
        auto_approve,
        sequential_approval,
        status,
        approval_levels
    } = req.body;

    try {
        // Update workflow
        await db.query(
            `UPDATE approval_workflows 
                SET name = ?, module = ?, operation = ?, description = ?, 
                    auto_approve = ?, sequential_approval = ?, status = ?
                WHERE id = ?`, [
                name,
                module,
                operation,
                description,
                auto_approve,
                sequential_approval,
                status,
                id
            ]
        );

        // Update approval levels if provided
        if (approval_levels && Array.isArray(approval_levels)) {
            // Delete existing levels
            await db.query('DELETE FROM approval_levels WHERE workflow_id = ?', [id]);

            // Insert new levels
            for (const level of approval_levels) {
                await db.query(
                    `INSERT INTO approval_levels 
                        (workflow_id, name, \`order\`, required, roles)
                        VALUES (?, ?, ?, ?, ?)`, [
                        id,
                        level.name,
                        level.order,
                        level.required !== undefined ? level.required : true,
                        JSON.stringify(level.roles || [])
                    ]
                );
            }
        }

        // Get updated workflow
        const [workflows] = await db.query(
            'SELECT * FROM approval_workflows WHERE id = ?', [id]
        );

        const [levels] = await db.query(
            'SELECT * FROM approval_levels WHERE workflow_id = ? ORDER BY `order`', [id]
        );

        // Parse JSON roles
        levels.forEach(level => {
            if (level.roles) {
                level.roles = JSON.parse(level.roles);
            }
        });

        const workflow = {...workflows[0], approval_levels: levels };

        res.json({
            success: true,
            message: 'Workflow updated successfully',
            data: workflow
        });
    } catch (error) {
        console.error('Error updating workflow:', error);
        res.status(500).json({ error: 'Failed to update workflow' });
    }
};

// Delete workflow
exports.deleteWorkflow = async(req, res) => {
    const { id } = req.params;

    try {
        // Check if workflow exists
        const [workflows] = await db.query(
            'SELECT id, status FROM approval_workflows WHERE id = ?', [id]
        );

        if (workflows.length === 0) {
            return res.status(404).json({ error: 'Workflow not found' });
        }

        // Check if there are pending approval requests
        const [requests] = await db.query(
            'SELECT COUNT(*) as count FROM approval_requests WHERE workflow_id = ? AND status IN ("pending", "under_review")', [id]
        );

        if (requests[0].count > 0) {
            return res.status(400).json({
                error: 'Cannot delete workflow with pending approval requests'
            });
        }

        // Delete approval levels
        await db.query('DELETE FROM approval_levels WHERE workflow_id = ?', [id]);

        // Delete workflow
        await db.query('DELETE FROM approval_workflows WHERE id = ?', [id]);

        res.json({
            success: true,
            message: 'Workflow deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting workflow:', error);
        res.status(500).json({ error: 'Failed to delete workflow' });
    }
};

// Get workflows by module and operation
exports.getWorkflowsByModule = async(req, res) => {
    const { module, operation } = req.query;

    try {
        let query = 'SELECT * FROM approval_workflows WHERE status = "active"';
        const params = [];

        if (module) {
            query += ' AND module = ?';
            params.push(module);
        }

        if (operation) {
            query += ' AND operation = ?';
            params.push(operation);
        }

        query += ' ORDER BY created_at DESC';

        const [workflows] = await db.query(query, params);

        // Get approval levels for each workflow
        for (let workflow of workflows) {
            const [levels] = await db.query(
                'SELECT * FROM approval_levels WHERE workflow_id = ? ORDER BY `order`', [workflow.id]
            );

            // Parse JSON roles for each level
            levels.forEach(level => {
                if (level.roles) {
                    try {
                        level.roles = JSON.parse(level.roles);
                    } catch (e) {
                        level.roles = [];
                    }
                }
            });

            workflow.approval_levels = levels;
        }

        res.json({
            success: true,
            data: workflows
        });
    } catch (error) {
        console.error('Error fetching workflows by module:', error);
        res.status(500).json({ error: 'Failed to fetch workflows' });
    }
};

// Get workflow statistics
exports.getWorkflowStats = async(req, res) => {
    try {
        const [stats] = await db.query(`
            SELECT 
                COUNT(*) as total_workflows,
                SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_workflows,
                SUM(CASE WHEN status = 'inactive' THEN 1 ELSE 0 END) as inactive_workflows,
                SUM(CASE WHEN status = 'draft' THEN 1 ELSE 0 END) as draft_workflows,
                SUM(CASE WHEN module = 'supply_chain' THEN 1 ELSE 0 END) as supply_chain_workflows,
                SUM(CASE WHEN module = 'finance' THEN 1 ELSE 0 END) as finance_workflows,
                SUM(CASE WHEN module = 'operations' THEN 1 ELSE 0 END) as operations_workflows,
                SUM(CASE WHEN auto_approve = 1 THEN 1 ELSE 0 END) as auto_approve_workflows,
                SUM(CASE WHEN sequential_approval = 1 THEN 1 ELSE 0 END) as sequential_approval_workflows
            FROM approval_workflows
        `);

        // Get pending approval requests count
        const [pendingStats] = await db.query(`
            SELECT COUNT(*) as pending_approvals
            FROM approval_requests 
            WHERE status IN ('pending', 'under_review')
        `);

        const finalStats = {
            ...stats[0],
            pending_approvals: pendingStats[0] ? pendingStats[0].pending_approvals || 0 : 0
        };

        res.json({
            success: true,
            data: finalStats
        });
    } catch (error) {
        console.error('Error fetching workflow stats:', error);
        res.status(500).json({ error: 'Failed to fetch workflow statistics' });
    }
};