const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const userModel = require('../models/userModel');
const userService = require('../services/userService');
const db = require('../config/db');

exports.register = async(req, res) => {
    // User registration should be handled by a superadmin via the userController
    res.status(403).json({ message: 'User registration is not allowed. Please contact an administrator.' });
};

exports.login = async(req, res) => {
    const { email, password } = req.body;

    try {
        const user = await userModel.findByEmail(email);

        // Detailed logging for user authentication
        if (!user) {
            console.log('No user found with email:', email);
            return res.status(401).json({ 
                message: 'Authentication failed. User not found.',
                details: { email }
            });
        }

        // Log user status details
        console.log('User status details:', {
            is_blocked: user.is_blocked,
            module_id: user.module_id,
            store_id: user.store_id
        });

        // Check if user is blocked
        if (user.is_blocked) {
            console.log('User authentication failed: User is blocked');
            return res.status(401).json({
                message: 'Authentication failed. User is blocked.',
                details: {
                    is_blocked: user.is_blocked
                }
            });
        }

        const isMatch = await bcrypt.compare(password, user.password);

        if (!isMatch) {
            return res.status(401).json({ message: "Authentication failed. Incorrect password." });
        }

        const token = jwt.sign({ id: user.id, role_id: user.role_id, module_id: user.module_id, store_id: user.store_id },
            process.env.AUTH_JWT_SECRET, { expiresIn: '24h' }
        );

        // Log the token creation and its expiration
        const decodedToken = jwt.decode(token);
        console.log(`Token created for user ${user.id}. Expires at: ${new Date(decodedToken.exp * 1000)}`);

        // Fetch roles and permissions
        const [roles] = await db.execute('SELECT name FROM roles WHERE id = ?', [user.role_id]);
        const [rolePermissions] = await db.execute('SELECT p.name FROM permissions p JOIN role_permissions rp ON p.id = rp.permission_id WHERE rp.role_id = ?', [user.role_id]);
        const [userPermissions] = await db.execute('SELECT p.name FROM permissions p JOIN user_permissions up ON p.id = up.permission_id WHERE up.user_id = ?', [user.id]);

        const permissions = [...new Set([...rolePermissions.map(p => p.name), ...userPermissions.map(p => p.name)])];

        console.log('Login successful. Token generated.');
        res.json({
            token,
            user: {
                id: user.id,
                name: user.name,
                email: user.email,
                module_id: user.module_id,
                module_name: user.module_name, // Include module_name in the response
                store_id: user.store_id,
                store_name: user.store_name, // Include store_name in the response
                roles: roles.map(r => r.name),
                permissions
            }
        });
    } catch (error) {
        console.error('Detailed login error:', {
            message: error.message,
            stack: error.stack,
            name: error.name
        });
        res.status(500).json({ 
            error: 'An unexpected error occurred during login.',
            details: error.message 
        });
    }
};

exports.getMe = async(req, res) => {
    try {
        const userDetails = await userService.getUserDetails(req.user.id);
        if (!userDetails) {
            return res.status(404).json({ message: "User not found" });
        }
        res.json(userDetails);
    } catch (error) {
        console.error('GetMe error:', error);
        res.status(500).json({ error: 'An unexpected error occurred while fetching user details.' });
    }
};

exports.logout = (req, res) => {
    res.json({ message: "User logged out successfully" });
};

// TODO: Consider moving user management functions (updateUser, blockUser, resetPassword, unblockUser)
// to userController with proper superAdminMiddleware protection for better separation of concerns.

exports.updateUser = async(req, res) => {
    const { id, name, email, role_id, module_id, username, store_id, phone_number } = req.body;

    try {
        const [result] = await db.execute(
            'UPDATE users SET name = ?, email = ?, role_id = ?, module_id = ?, username = ?, store_id = ?, phone_number = ? WHERE id = ?', [name, email, role_id, module_id, username, store_id, phone_number, id]
        );

        if (result.affectedRows === 0) return res.status(404).json({ message: 'User not found' });

        res.json({ message: 'User updated successfully' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

exports.blockUser = async(req, res) => {
    const { id } = req.params;

    try {
        const [result] = await db.execute(
            'UPDATE users SET is_blocked = 1 WHERE id = ?', [id]
        );

        if (result.affectedRows === 0) return res.status(404).json({ message: 'User not found' });

        res.json({ message: 'User blocked successfully' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Reset user password
exports.resetPassword = async(req, res) => {
    const { id } = req.body;
    const newPassword = "hybrid2025";
    const hashedPassword = await bcrypt.hash(newPassword, 10);

    try {
        const [result] = await db.execute(
            "UPDATE users SET password = ? WHERE id = ?", [hashedPassword, id]
        );

        if (result.affectedRows === 0) return res.status(404).json({ message: 'User not found' });

        res.json({ message: 'Password reset successfully' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Unblock a user
exports.unblockUser = async(req, res) => {
    const { id } = req.params;

    try {
        const [result] = await db.execute(
            "UPDATE users SET is_blocked = 0 WHERE id = ?", [id]
        );

        if (result.affectedRows === 0) return res.status(404).json({ message: 'User not found' });

        res.json({ message: 'User unblocked successfully' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Validate token
exports.validateToken = async (req, res) => {
    try {
        const token = req.headers.authorization && req.headers.authorization.split(' ')[1];

        if (!token) {
            return res.status(401).json({ success: false, message: 'No token provided.' });
        }

        jwt.verify(token, process.env.AUTH_JWT_SECRET, (err, decoded) => {
            if (err) {
                return res.status(401).json({ success: false, message: 'Invalid token.', error: err.message });
            }
            // Token is valid, return user data
            res.json({ success: true, user: decoded });
        });
    } catch (error) {
        console.error('Error validating token:', error);
        res.status(500).json({ success: false, message: 'Server error during token validation.' });
    }
};
