const ChatModel = require('../models/chatModel');
const FileUploadService = require('../services/fileUploadService');
const jwt = require('jsonwebtoken');
const multer = require('multer');
const path = require('path');
const fsp = require('fs').promises; // Use fs.promises for async operations
const fs = require('fs'); // Still need synchronous fs for existsSync for now

// Configure multer for file uploads
const storage = multer.memoryStorage();
const upload = multer({
    storage: storage,
    limits: {
        fileSize: 10 * 1024 * 1024, // 10MB limit
        files: 1 // Only one file at a time
    },
    fileFilter: (req, file, cb) => {
        // Validate file type
        const allowedTypes = [
            'image/jpeg', 'image/png', 'image/gif', 'image/webp',
            'application/pdf', 'text/plain', 'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        ];

        if (allowedTypes.includes(file.mimetype)) {
            cb(null, true);
        } else {
            cb(new Error('Invalid file type'), false);
        }
    }
});

class ChatController {
    // Get user's conversations
    static async getUserConversations(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const userId = decoded.id;

            const conversations = await ChatModel.getUserConversations(userId);

            res.json({
                success: true,
                data: conversations
            });
        } catch (error) {
            console.error('Error getting user conversations:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Create new conversation
    static async createConversation(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const userId = decoded.id;

            const { participants, title, type } = req.body;

            if (!participants || !Array.isArray(participants) || participants.length === 0) {
                return res.status(400).json({ error: 'Participants array is required' });
            }

            // Ensure the creator is included in participants
            if (!participants.includes(userId)) {
                participants.push(userId);
            }

            const conversationId = await ChatModel.createConversation(participants, title, type);

            // Get the created conversation details
            const conversations = await ChatModel.getUserConversations(userId);
            const conversation = conversations.find(c => c.id === conversationId);

            if (!conversation) {
                throw new Error('Failed to retrieve created conversation');
            }

            res.json({
                success: true,
                message: 'Conversation created successfully',
                data: conversation
            });
        } catch (error) {
            console.error('Error creating conversation:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Get conversation messages
    static async getConversationMessages(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const userId = decoded.id;

            const { conversationId } = req.params;
            const { limit = 50, offset = 0 } = req.query;

            // Check if user is participant in conversation
            const isParticipant = await ChatModel.isUserInConversation(userId, conversationId);
            if (!isParticipant) {
                return res.status(403).json({ error: 'Access denied' });
            }

            const messages = await ChatModel.getConversationMessages(conversationId, userId, parseInt(limit), parseInt(offset));

            // Mark messages as read
            await ChatModel.markMessagesAsRead(userId, conversationId);

            res.json({
                success: true,
                data: messages
            });
        } catch (error) {
            console.error('Error getting conversation messages:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Get conversation participants
    static async getConversationParticipants(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const userId = decoded.id;

            const { conversationId } = req.params;

            // Check if user is participant in conversation
            const isParticipant = await ChatModel.isUserInConversation(userId, conversationId);
            if (!isParticipant) {
                return res.status(403).json({ error: 'Access denied' });
            }

            const participants = await ChatModel.getConversationParticipants(conversationId);

            res.json({
                success: true,
                data: participants
            });
        } catch (error) {
            console.error('Error getting conversation participants:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Send message (this will be handled by WebSocket, but keeping for REST fallback)
    static async sendMessage(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const senderId = decoded.id;

            const { conversationId, message, messageType = 'text' } = req.body;

            if (!message || !conversationId) {
                return res.status(400).json({ error: 'Message and conversationId are required' });
            }

            // Check if user is participant in conversation
            const isParticipant = await ChatModel.isUserInConversation(senderId, conversationId);
            if (!isParticipant) {
                return res.status(403).json({ error: 'Access denied' });
            }

            const messageId = await ChatModel.saveMessage(conversationId, senderId, message, messageType);

            res.json({
                success: true,
                data: { messageId }
            });
        } catch (error) {
            console.error('Error sending message:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Get unread message count
    static async getUnreadCount(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const userId = decoded.id;

            const { conversationId } = req.params;

            // Check if user is participant in conversation
            const isParticipant = await ChatModel.isUserInConversation(userId, conversationId);
            if (!isParticipant) {
                return res.status(403).json({ error: 'Access denied' });
            }

            const unreadCount = await ChatModel.getUnreadCount(userId, conversationId);

            res.json({
                success: true,
                data: { unreadCount }
            });
        } catch (error) {
            console.error('Error getting unread count:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Mark messages as read
    static async markMessagesAsRead(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const userId = decoded.id;

            const { conversationId } = req.params;

            // Check if user is participant in conversation
            const isParticipant = await ChatModel.isUserInConversation(userId, conversationId);
            if (!isParticipant) {
                return res.status(403).json({ error: 'Access denied' });
            }

            // Mark messages as read
            await ChatModel.markMessagesAsRead(userId, conversationId);

            res.json({
                success: true,
                message: 'Messages marked as read'
            });
        } catch (error) {
            console.error('Error marking messages as read:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Get online users
    static async getOnlineUsers(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const currentUserId = decoded.id;

            // Get all users from the database with online status
            const db = require('../config/db');
            const [allUsers] = await db.execute(`
                SELECT id, name, username, email, is_online, last_seen_at
                FROM users
                ORDER BY is_online DESC, last_seen_at DESC
            `);

            // Filter out the current user and format the response
            const otherUsers = allUsers
                .filter(user => user.id !== currentUserId)
                .map(user => ({
                    id: user.id,
                    name: user.name,
                    username: user.username || user.email, // Use username if available, otherwise email
                    email: user.email,
                    isOnline: user.is_online === 1, // Convert MySQL BOOLEAN (TINYINT) to boolean
                    lastSeen: user.last_seen_at
                }));

            res.json({
                success: true,
                data: otherUsers
            });
        } catch (error) {
            console.error('Error getting online users:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Update user status
    static async updateUserStatus(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const userId = decoded.id;
            const { status } = req.body;

            // In a real implementation, you would:
            // 1. Update the user's online status in the database
            // 2. Update Redis or similar for real-time status
            // 3. Broadcast status change to other users via WebSocket

            console.log(`User ${userId} status updated to: ${status}`);

            res.json({
                success: true,
                message: 'Status updated successfully'
            });
        } catch (error) {
            console.error('Error updating user status:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Get user status
    static async getUserStatus(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const { userId } = req.params;

            // Get user from database with online status
            const db = require('../config/db');
            const [users] = await db.execute(`
                SELECT id, name, username, email, is_online, last_seen_at
                FROM users
                WHERE id = ?
            `, [userId]);

            if (users.length === 0) {
                return res.status(404).json({ error: 'User not found' });
            }

            const user = users[0];
            const userStatus = {
                id: user.id,
                name: user.name,
                username: user.username || user.email,
                isOnline: user.is_online === 1, // Convert MySQL BOOLEAN (TINYINT) to boolean
                lastSeen: user.last_seen_at
            };

            res.json({
                success: true,
                data: userStatus
            });
        } catch (error) {
            console.error('Error getting user status:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    static async searchUsers(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const currentUserId = decoded.id;

            const { q: searchQuery, limit = 20, offset = 0 } = req.query;

            const db = require('../config/db');

            let query = `
                SELECT
                    id,
                    name,
                    username,
                    email,
                    is_online,
                    last_seen_at,
                    created_at
                FROM users
                WHERE id != ?
            `;

            let params = [currentUserId];

            // Add search conditions if query provided
            if (searchQuery && searchQuery.trim()) {
                const searchTerm = `%${searchQuery.trim()}%`;
                query += ` AND (
                    name LIKE ? OR
                    username LIKE ? OR
                    email LIKE ?
                )`;
                params.push(searchTerm, searchTerm, searchTerm);
            }

            // Add ordering - prioritize online users, then by last seen, then by name
            query += ` ORDER BY
                is_online DESC,
                last_seen_at DESC,
                name ASC
            `;

            // Add pagination
            query += ` LIMIT ? OFFSET ?`;
            params.push(parseInt(limit), parseInt(offset));

            const [users] = await db.execute(query, params);

            // Format the response data
            const formattedUsers = users.map(user => ({
                id: user.id,
                name: user.name,
                username: user.username,
                email: user.email,
                isOnline: user.is_online === 1,
                lastSeen: user.last_seen_at,
                createdAt: user.created_at
            }));

            // Get total count for pagination (without LIMIT/OFFSET)
            let countQuery = `
                SELECT COUNT(*) as total
                FROM users
                WHERE id != ?
            `;

            let countParams = [currentUserId];

            if (searchQuery && searchQuery.trim()) {
                const searchTerm = `%${searchQuery.trim()}%`;
                countQuery += ` AND (
                    name LIKE ? OR
                    username LIKE ? OR
                    email LIKE ?
                )`;
                countParams.push(searchTerm, searchTerm, searchTerm);
            }

            const [countResult] = await db.execute(countQuery, countParams);
            const totalCount = countResult[0].total;

            res.json({
                success: true,
                data: formattedUsers,
                pagination: {
                    total: totalCount,
                    limit: parseInt(limit),
                    offset: parseInt(offset),
                    hasMore: parseInt(offset) + formattedUsers.length < totalCount
                }
            });

        } catch (error) {
            console.error('Error searching users:', error);
            res.status(500).json({
                error: 'Internal server error',
                message: error.message
            });
        }
    }

    // File upload endpoint
    static async uploadFile(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const userId = decoded.id;

            if (!req.file) {
                return res.status(400).json({ error: 'No file provided' });
            }

            // Validate file
            const validation = FileUploadService.validateFile(req.file);
            if (!validation.isValid) {
                return res.status(400).json({
                    error: 'File validation failed',
                    details: validation.errors
                });
            }

            // Save file
            const fileInfo = await FileUploadService.saveFile(
                req.file.buffer,
                req.file.originalname,
                req.file.mimetype
            );

            res.json({
                success: true,
                data: {
                    ...fileInfo,
                    uploadedBy: userId
                }
            });

        } catch (error) {
            console.error('Error uploading file:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Send file message
    static async sendFileMessage(req, res) {
        try {
            const token = req.headers.authorization ? req.headers.authorization.split(' ')[1] : null;
            if (!token) {
                return res.status(401).json({ error: 'No token provided' });
            }

            const decoded = jwt.verify(token, process.env.JWT_SECRET);
            const userId = decoded.id;

            const {
                conversationId,
                message,
                fileName,
                fileSize,
                fileUrl,
                filePath,
                fileMimeType,
                fileThumbnailPath,
                fileThumbnailUrl
            } = req.body;

            if (!conversationId) {
                return res.status(400).json({ error: 'Conversation ID is required' });
            }

            // Check if user is participant in conversation
            const isParticipant = await ChatModel.isUserInConversation(userId, conversationId);
            if (!isParticipant) {
                return res.status(403).json({ error: 'Access denied to conversation' });
            }

            // Determine message type
            const messageType = fileMimeType.startsWith('image/') ? 'image' : 'file';

            // Save message to database
            const messageId = await ChatModel.saveFileMessage({
                conversationId,
                senderId: userId,
                message: message || '',
                messageType,
                fileName,
                fileSize,
                fileUrl,
                filePath,
                fileMimeType,
                fileThumbnailPath
            });

            // Get message details for response
            const messageData = await ChatModel.getMessageById(messageId);

            res.json({
                success: true,
                data: messageData
            });

        } catch (error) {
            console.error('Error sending file message:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Serve uploaded files
    static async serveFile(req, res) {
        try {
            let fileName = req.params.fileName;

            // Check if this is a thumbnail request by checking the route
            const isThumbnail = req.route.path.includes('/thumbnails');
            let filePath;

            console.log(`Serving file request:`, {
                fileName,
                route: req.route.path,
                isThumbnail,
                fullPath: req.path,
                originalUrl: req.originalUrl
            });

            if (isThumbnail) {
                // For thumbnails: /files/thumbnails/filename.jpg
                filePath = path.join(__dirname, '..', 'uploads', 'thumbnails', fileName);
            } else {
                // For regular files: /files/filename.jpg
                filePath = path.join(__dirname, '..', 'uploads', fileName);
            }

            console.log(`File path: ${filePath}`);

            if (!fs.existsSync(filePath)) {
                console.warn(`File not found at path: ${filePath}`);
                return res.status(404).json({ error: 'File not found' });
            }

            const stat = await fsp.stat(filePath);
            if (!stat.isFile()) {
                console.warn(`Path is not a file: ${filePath}`);
                return res.status(400).json({ error: 'Invalid file path' });
            }

            console.log('File exists and is a file.');

            // For chat files, allow unauthenticated access since they're protected by conversation access
            // TODO: Add proper conversation-based access control later
            console.log(`Serving file: ${fileName}`);

            // Set appropriate headers
            const mimeTypes = {
                '.jpg': 'image/jpeg',
                '.jpeg': 'image/jpeg',
                '.png': 'image/png',
                '.gif': 'image/gif',
                '.webp': 'image/webp',
                '.pdf': 'application/pdf',
                '.doc': 'application/msword',
                '.docx': 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                '.xls': 'application/vnd.ms-excel',
                '.xlsx': 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                '.txt': 'text/plain',
                '.zip': 'application/zip',
                '.mp3': 'audio/mpeg',
                '.mp4': 'video/mp4',
            };

            const ext = path.extname(fileName).toLowerCase();
            const mimeType = mimeTypes[ext] || 'application/octet-stream';
            res.setHeader('Content-Type', mimeType);
            console.log(`Setting Content-Type: ${mimeType}`);

            // For images and PDFs, allow inline display
            if (mimeType.startsWith('image/') || mimeType === 'application/pdf') {
                res.setHeader('Content-Disposition', 'inline');
                console.log('Setting Content-Disposition: inline');
            } else {
                res.setHeader('Content-Disposition', `attachment; filename="${fileName}"`);
                console.log(`Setting Content-Disposition: attachment; filename="${fileName}"`);
            }

            // Stream file to response
            console.log('Creating read stream...');
            const fileStream = fs.createReadStream(filePath);
            fileStream.on('error', (streamErr) => {
                console.error('File stream error:', streamErr);
                if (!res.headersSent) {
                    res.status(500).json({ error: 'File streaming error' });
                }
            });
            fileStream.pipe(res);
            console.log('File stream piped to response.');

        } catch (error) {
            console.error('Error serving file:', error);
            res.status(500).json({ error: 'Internal server error' });
        }
    }

    // Get multer upload middleware
    static getUploadMiddleware() {
        return upload.single('file');
    }
}

module.exports = ChatController;