const db = require('../config/db');

// Get all CRUD operations
exports.getAllOperations = async(req, res) => {
    try {
        const [operations] = await db.query(`
            SELECT 
                cor.*,
                aw.name as workflow_name,
                aw.module as workflow_module,
                aw.operation as workflow_operation
            FROM crud_operations_registry cor
            LEFT JOIN approval_workflows aw ON cor.approval_workflow_id = aw.id
            ORDER BY cor.module, cor.operation
        `);

        res.json({
            success: true,
            data: operations
        });
    } catch (error) {
        console.error('Error fetching CRUD operations:', error);
        res.status(500).json({ error: 'Failed to fetch CRUD operations' });
    }
};

// Get CRUD operations by module
exports.getOperationsByModule = async(req, res) => {
    try {
        const { module } = req.params;

        const [operations] = await db.query(`
            SELECT 
                cor.*,
                aw.name as workflow_name,
                aw.module as workflow_module,
                aw.operation as workflow_operation
            FROM crud_operations_registry cor
            LEFT JOIN approval_workflows aw ON cor.approval_workflow_id = aw.id
            WHERE cor.module = ?
            ORDER BY cor.operation
        `, [module]);

        res.json({
            success: true,
            data: operations
        });
    } catch (error) {
        console.error('Error fetching CRUD operations by module:', error);
        res.status(500).json({ error: 'Failed to fetch CRUD operations' });
    }
};

// Get CRUD operation by ID
exports.getOperationById = async(req, res) => {
    try {
        const { id } = req.params;

        const [operations] = await db.query(`
            SELECT 
                cor.*,
                aw.name as workflow_name,
                aw.module as workflow_module,
                aw.operation as workflow_operation
            FROM crud_operations_registry cor
            LEFT JOIN approval_workflows aw ON cor.approval_workflow_id = aw.id
            WHERE cor.id = ?
        `, [id]);

        if (operations.length === 0) {
            return res.status(404).json({ error: 'CRUD operation not found' });
        }

        res.json({
            success: true,
            data: operations[0]
        });
    } catch (error) {
        console.error('Error fetching CRUD operation:', error);
        res.status(500).json({ error: 'Failed to fetch CRUD operation' });
    }
};

// Create new CRUD operation
exports.createOperation = async(req, res) => {
    try {
        const {
            module,
            operation,
            entity_type,
            requires_approval,
            approval_workflow_id,
            description
        } = req.body;

        if (!module || !operation || !entity_type) {
            return res.status(400).json({
                error: 'Missing required fields',
                required: ['module', 'operation', 'entity_type']
            });
        }

        // Check if operation already exists
        const [existing] = await db.query(`
            SELECT id FROM crud_operations_registry 
            WHERE module = ? AND operation = ?
        `, [module, operation]);

        if (existing.length > 0) {
            return res.status(400).json({
                error: 'Operation already exists',
                message: `Operation ${module}.${operation} is already registered`
            });
        }

        const [result] = await db.query(`
            INSERT INTO crud_operations_registry 
            (module, operation, entity_type, requires_approval, approval_workflow_id, description, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        `, [
            module,
            operation,
            entity_type,
            requires_approval || false,
            approval_workflow_id || null,
            description || null,
            req.headers['user-id'] || null
        ]);

        res.status(201).json({
            success: true,
            message: 'CRUD operation registered successfully',
            data: { id: result.insertId }
        });
    } catch (error) {
        console.error('Error creating CRUD operation:', error);
        res.status(500).json({ error: 'Failed to create CRUD operation' });
    }
};

// Update CRUD operation
exports.updateOperation = async(req, res) => {
    try {
        const { id } = req.params;
        const {
            module,
            operation,
            entity_type,
            requires_approval,
            approval_workflow_id,
            description
        } = req.body;

        // Check if operation exists
        const [existing] = await db.query(`
            SELECT id FROM crud_operations_registry WHERE id = ?
        `, [id]);

        if (existing.length === 0) {
            return res.status(404).json({ error: 'CRUD operation not found' });
        }

        await db.query(`
            UPDATE crud_operations_registry 
            SET module = ?, operation = ?, entity_type = ?, requires_approval = ?, 
                approval_workflow_id = ?, description = ?, updated_at = NOW()
            WHERE id = ?
        `, [
            module,
            operation,
            entity_type,
            requires_approval,
            approval_workflow_id,
            description,
            id
        ]);

        res.json({
            success: true,
            message: 'CRUD operation updated successfully'
        });
    } catch (error) {
        console.error('Error updating CRUD operation:', error);
        res.status(500).json({ error: 'Failed to update CRUD operation' });
    }
};

// Delete CRUD operation
exports.deleteOperation = async(req, res) => {
    try {
        const { id } = req.params;

        // Check if operation exists
        const [existing] = await db.query(`
            SELECT id FROM crud_operations_registry WHERE id = ?
        `, [id]);

        if (existing.length === 0) {
            return res.status(404).json({ error: 'CRUD operation not found' });
        }

        await db.query('DELETE FROM crud_operations_registry WHERE id = ?', [id]);

        res.json({
            success: true,
            message: 'CRUD operation deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting CRUD operation:', error);
        res.status(500).json({ error: 'Failed to delete CRUD operation' });
    }
};

// Bulk import CRUD operations
exports.bulkImportOperations = async(req, res) => {
    try {
        const { operations } = req.body;

        if (!Array.isArray(operations)) {
            return res.status(400).json({
                error: 'Invalid operations array'
            });
        }

        const results = [];
        const errors = [];

        for (const operation of operations) {
            try {
                const {
                    module,
                    operation: operationName,
                    entity_type,
                    requires_approval,
                    approval_workflow_id,
                    description
                } = operation;

                if (!module || !operationName || !entity_type) {
                    errors.push({
                        operation: operationName,
                        error: 'Missing required fields'
                    });
                    continue;
                }

                // Check if operation already exists
                const [existing] = await db.query(`
                    SELECT id FROM crud_operations_registry 
                    WHERE module = ? AND operation = ?
                `, [module, operationName]);

                if (existing.length > 0) {
                    errors.push({
                        operation: operationName,
                        error: 'Operation already exists'
                    });
                    continue;
                }

                const [result] = await db.query(`
                    INSERT INTO crud_operations_registry 
                    (module, operation, entity_type, requires_approval, approval_workflow_id, description, created_by)
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                `, [
                    module,
                    operationName,
                    entity_type,
                    requires_approval || false,
                    approval_workflow_id || null,
                    description || null,
                    req.headers['user-id'] || null
                ]);

                results.push({
                    id: result.insertId,
                    operation: operationName,
                    status: 'created'
                });

            } catch (error) {
                errors.push({
                    operation: operation.operation,
                    error: error.message
                });
            }
        }

        res.json({
            success: true,
            message: `Bulk import completed. ${results.length} created, ${errors.length} failed`,
            data: {
                created: results,
                errors: errors
            }
        });
    } catch (error) {
        console.error('Error bulk importing CRUD operations:', error);
        res.status(500).json({ error: 'Failed to bulk import CRUD operations' });
    }
};

// Get operations statistics
exports.getOperationsStats = async(req, res) => {
    try {
        // Get total operations by module
        const [moduleStats] = await db.query(`
            SELECT 
                module,
                COUNT(*) as total_operations,
                SUM(CASE WHEN requires_approval = 1 THEN 1 ELSE 0 END) as operations_requiring_approval,
                SUM(CASE WHEN requires_approval = 0 THEN 1 ELSE 0 END) as operations_no_approval
            FROM crud_operations_registry
            GROUP BY module
            ORDER BY module
        `);

        // Get operations by approval status
        const [approvalStats] = await db.query(`
            SELECT 
                requires_approval,
                COUNT(*) as count
            FROM crud_operations_registry
            GROUP BY requires_approval
        `);

        // Get operations by workflow
        const [workflowStats] = await db.query(`
            SELECT 
                aw.name as workflow_name,
                COUNT(cor.id) as operation_count
            FROM approval_workflows aw
            LEFT JOIN crud_operations_registry cor ON aw.id = cor.approval_workflow_id
            GROUP BY aw.id, aw.name
            ORDER BY operation_count DESC
        `);

        res.json({
            success: true,
            data: {
                module_stats: moduleStats,
                approval_stats: approvalStats,
                workflow_stats: workflowStats
            }
        });
    } catch (error) {
        console.error('Error fetching CRUD operations stats:', error);
        res.status(500).json({ error: 'Failed to fetch CRUD operations stats' });
    }
};