const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const superAdminModel = require('../models/superAdminModel');

// Super admin login
exports.login = async(req, res) => {
    const { username, password } = req.body;
    
    console.log('Login attempt:', { username });

    try {
        // Check if admin exists
        const admin = await superAdminModel.getSuperAdminByUsername(username);
        console.log('Admin found:', !!admin);
        
        if (!admin) {
            return res.status(401).json({ message: 'Invalid credentials' });
        }

        // Check if admin is active
        console.log('Admin active:', admin.is_active);
        if (!admin.is_active) {
            return res.status(403).json({ message: 'Account is deactivated' });
        }

        // Verify password
        const isPasswordValid = await bcrypt.compare(password, admin.password);
        console.log('Password valid:', isPasswordValid);
        console.log('Password in DB:', admin.password);
        
        if (!isPasswordValid) {
            return res.status(401).json({ message: 'Invalid credentials' });
        }

        // Update last login time
        await superAdminModel.updateLastLogin(admin.id);

        const token = jwt.sign(
            { user: { id: admin.id, username: admin.username, isSuperAdmin: true } },
            process.env.AUTH_JWT_SECRET,
            { expiresIn: '1h' }
        );

        // Return token and admin data
        res.json({ token });
    } catch (error) {
        console.error('Super admin login error:', error);
        res.status(500).json({ message: 'Server error', error: error.message });
    }
};

// Get current super admin profile
exports.getProfile = async(req, res) => {
    try {
        const admin = await superAdminModel.getSuperAdminById(req.user.id);
        if (!admin) {
            return res.status(404).json({ message: 'Super admin not found' });
        }

        res.json({
            id: admin.id,
            name: admin.name,
            email: admin.email,
            username: admin.username,
            lastLogin: admin.last_login
        });
    } catch (error) {
        console.error('Get profile error:', error);
        res.status(500).json({ message: 'Server error', error: error.message });
    }
};

// Update super admin profile
exports.updateProfile = async(req, res) => {
    const { name, email } = req.body;

    try {
        await superAdminModel.updateSuperAdmin(req.user.id, name, email);
        res.json({ message: 'Profile updated successfully' });
    } catch (error) {
        console.error('Update profile error:', error);
        res.status(500).json({ message: 'Server error', error: error.message });
    }
};

// Change password
exports.changePassword = async(req, res) => {
    const { currentPassword, newPassword } = req.body;

    try {
        // Get admin
        const admin = await superAdminModel.getSuperAdminById(req.user.id);
        if (!admin) {
            return res.status(404).json({ message: 'Super admin not found' });
        }

        // Verify current password
        const isPasswordValid = await bcrypt.compare(currentPassword, admin.password);
        if (!isPasswordValid) {
            return res.status(401).json({ message: 'Current password is incorrect' });
        }

        // Hash new password
        const hashedPassword = await bcrypt.hash(newPassword, 10);

        // Update password
        await superAdminModel.changePassword(req.user.id, hashedPassword);

        res.json({ message: 'Password changed successfully' });
    } catch (error) {
        console.error('Change password error:', error);
        res.status(500).json({ message: 'Server error', error: error.message });
    }
};