const bcrypt = require('bcryptjs');
const userModel = require('../models/userModel');
const db = require('../config/db');
const { faker } = require('@faker-js/faker');

// Get all users
exports.getAllUsers = async(req, res) => {
    try {
        const users = await userModel.getAllUsers();
        res.json(users);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Get a single user by ID
exports.getUserById = async(req, res) => {
    try {
        const user = await userModel.getUserById(req.params.id);
        if (!user) return res.status(404).json({ message: "User not found" });
        res.json(user);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Create a new user
exports.createUser = async(req, res) => {
    try {
        const { name, email, username, role, module, phone_number, store_id } = req.body;

        // Generate a temporary password
        const tempPassword = faker.internet.password();
        const hashedPassword = await bcrypt.hash(tempPassword, 10);

        const userId = await userModel.createUser(name, email, hashedPassword, username, role, module, phone_number, store_id);

        // Here, you would trigger an email/SMS/WhatsApp message with the tempPassword
        // For now, we'll just log it to the console
        console.log(`Password for ${email}: ${tempPassword}`);

        res.status(201).json({ message: "User created successfully", userId });
    } catch (error) {
        console.error("Error creating user:", error);
        res.status(500).json({ error: error.message });
    }
};

// Update user details
exports.updateUser = async(req, res) => {
    try {
        const { name, email, role, module, phone_number, store_id } = req.body;
        await userModel.updateUser(req.params.id, name, email, role, module, phone_number, store_id);
        res.json({ message: "User updated successfully" });
    } catch (error) {
        console.error("Error updating user:", error);
        res.status(500).json({ error: error.message });
    }
};

// Delete a user
exports.deleteUser = async(req, res) => {
    try {
        await userModel.deleteUser(req.params.id);
        res.json({ message: "User deleted successfully" });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Change Password
exports.changePassword = async(req, res) => {
    try {
        const { oldPassword, newPassword } = req.body;
        const userId = req.params.id;

        if (!userId) {
            return res.status(400).json({ message: "User ID is required in the URL" });
        }

        // Get user's current password
        const [users] = await db.execute("SELECT password FROM users WHERE id = ?", [userId]);
        if (users.length === 0) {
            return res.status(404).json({ message: "User not found" });
        }

        const user = users[0];

        // Verify old password
        const isMatch = await bcrypt.compare(oldPassword, user.password);
        if (!isMatch) {
            return res.status(400).json({ message: "Old password is incorrect" });
        }

        // Hash new password
        const hashedPassword = await bcrypt.hash(newPassword, 10);

        // Update password in the database
        await db.execute("UPDATE users SET password = ? WHERE id = ?", [hashedPassword, userId]);

        res.json({ message: "Password changed successfully" });
    } catch (error) {
        console.error("Error changing password:", error);
        res.status(500).json({ error: error.message });
    }
};

// Block a user
exports.blockUser = async(req, res) => {
    try {
        const [result] = await db.execute("UPDATE users SET is_blocked = 1, is_active = 0 WHERE id = ?", [req.params.id]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ message: "User not found" });
        }
        res.json({ message: "User blocked successfully" });
    } catch (error) {
        console.error("Error blocking user:", error);
        res.status(500).json({ error: error.message });
    }
};

// Unblock a user
exports.unblockUser = async(req, res) => {
    try {
        const [result] = await db.execute("UPDATE users SET is_blocked = 0, is_active = 1 WHERE id = ?", [req.params.id]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ message: "User not found" });
        }
        res.json({ message: "User unblocked successfully" });
    } catch (error) {
        console.error("Error unblocking user:", error);
        res.status(500).json({ error: error.message });
    }
};

// Reset user password
exports.resetPassword = async(req, res) => {
    try {
        const newPassword = "hybrid2025"; // Or generate a random one
        const hashedPassword = await bcrypt.hash(newPassword, 10);
        const [result] = await db.execute("UPDATE users SET password = ? WHERE id = ?", [hashedPassword, req.params.id]);

        if (result.affectedRows === 0) {
            return res.status(404).json({ message: "User not found" });
        }
        res.json({ message: "Password reset successfully" });
    } catch (error) {
        console.error("Error resetting password:", error);
        res.status(500).json({ error: error.message });
    }
};

// Get usernames by IDs
exports.getUsernames = async(req, res) => {
    try {
        const { userIds } = req.body;
        console.log('userController.getUsernames: Received userIds:', userIds); // Log incoming user IDs

        if (!userIds || !Array.isArray(userIds) || userIds.length === 0) {
            console.warn('userController.getUsernames: Invalid or empty userIds array.');
            return res.status(400).json({ success: false, message: 'userIds array is required' });
        }

        const users = await userModel.getUsersByIds(userIds);
        console.log('userController.getUsernames: Users from model:', users); // Log users from model
        const usernamesMap = {};
        users.forEach(user => {
            usernamesMap[user.id] = user.username;
        });
        console.log('userController.getUsernames: Usernames map:', usernamesMap); // Log final map

        res.json({ success: true, data: usernamesMap });
    } catch (error) {
        console.error("userController.getUsernames error:", error); // Log error
        res.status(500).json({ success: false, message: "Failed to fetch usernames", error: error.message });
    }
};