const db = require('../config/db');
const axios = require('axios');
const crypto = require('crypto');
const operationModel = require('../models/operationModel');
const { emitToRoom } = require('../socket');
const notificationService = require('../services/notificationService');
const createOperationsRegistryTables = require('../scripts/createOperationsRegistryTables');
const createCentralWorkflowTables = require('../scripts/createCentralWorkflowTables');
const migrateApprovalTablesForPayloads = require('../scripts/migrate_approval_tables_for_payloads');

console.log('🔍 WorkflowController module loaded');

const ROLE_NAME_FALLBACKS = {
    manager: 'Manager',
    cfo: 'Chief Financial Officer',
    ceo: 'Chief Executive Officer',
    finance: 'Finance Officer',
    accountant: 'Accountant',
    supervisor: 'Supervisor',
    director: 'Director',
    vp: 'Vice President',
    hr: 'Human Resources',
    it: 'IT Administrator',
    admin: 'Administrator',
    super_admin: 'Super Administrator',
    user: 'User',
    approver: 'Approver'
};

// Helper function to log system actions in approval timeline
async function logSystemAction(requestId, stepId, action, actorName = 'System', comment = '') {
    try {
        await db.query(
            `INSERT INTO approval_actions (request_id, step_id, actor_id, actor_name, action, comment)
             VALUES (?, ?, 0, ?, ?, ?)`,
            [requestId, stepId || null, actorName, action, comment]
        );
    } catch (error) {
        console.error('Error logging system action:', error);
    }
}

// Helper function to create standardized workflow notifications
async function notifyWorkflowEvent(userId, eventType, data) {
    try {
        let title, message, link, type, priority, icon;

        switch (eventType) {
            case 'request_created':
                title = 'New Approval Request';
                message = `${data.module} - ${data.operation}: ${data.title}`;
                link = '/dashboard/pending-approvals';
                type = 'info';
                priority = 'normal';
                icon = 'assignment';
                break;

            case 'approval_needed':
                title = 'Approval Required';
                message = `${data.module} - ${data.operation}: ${data.title}`;
                link = '/dashboard/pending-approvals';
                type = 'warning';
                priority = 'high';
                icon = 'pending_actions';
                break;

            case 'request_approved':
                title = 'Request Approved';
                message = `${data.module} - ${data.operation}: ${data.title}`;
                link = '/dashboard/pending-approvals';
                type = 'success';
                priority = 'normal';
                icon = 'check_circle';
                break;

            case 'request_rejected':
                title = 'Request Rejected';
                message = `${data.module} - ${data.operation}: ${data.title}`;
                link = '/dashboard/pending-approvals';
                type = 'error';
                priority = 'normal';
                icon = 'cancel';
                break;

            case 'step_completed':
                title = 'Workflow Step Completed';
                message = `${data.stepName}: ${data.module} - ${data.operation}`;
                link = '/dashboard/pending-approvals';
                type = 'info';
                priority = 'low';
                icon = 'done_all';
                break;

            case 'workflow_completed':
                title = 'Workflow Completed';
                message = `${data.module} - ${data.operation}: ${data.title} has been fully approved`;
                link = '/dashboard/pending-approvals';
                type = 'success';
                priority = 'normal';
                icon = 'check_circle';
                break;

            default:
                title = 'Workflow Update';
                message = `${data.module} - ${data.operation}: ${data.title}`;
                link = '/dashboard/pending-approvals';
                type = 'info';
                priority = 'normal';
                icon = 'notifications';
        }

        await notificationService.createNotification({
            userId,
            title,
            message,
            link,
            type,
            priority,
            icon,
            sourceModule: 'workflow',
            payload: data
        });

        console.log(`📬 Workflow notification sent to user ${userId}: ${eventType}`);
    } catch (error) {
        console.error('Failed to create workflow notification:', error);
    }
}

function createAssigneeCache() {
    return {
        users: new Map(),
        rolesById: new Map(),
        rolesByName: new Map()
    };
}

function normalizeAssigneeEntry(assignee) {
    if (!assignee) return null;

    if (typeof assignee === 'string') {
        const [typeRaw, ...rest] = assignee.split(':');
        const ref = rest.length ? rest.join(':') : typeRaw;
        return {
            type: (rest.length ? typeRaw : 'user').toLowerCase(),
            refId: ref
        };
    }

    const type = (assignee.type || 'user').toLowerCase();
    let refId = assignee.refId ?? assignee.id ?? assignee.value ?? assignee.identifier;

    if (!refId && assignee.label && assignee.label.includes(':')) {
        const [, ref] = assignee.label.split(':');
        refId = ref;
    }

    if (refId === undefined || refId === null) return null;

    return {
        ...assignee,
        type,
        refId
    };
}

function normalizeId(value) {
    if (value === undefined || value === null) return '';
    return String(value);
}

function isNumeric(value) {
    return /^\d+$/.test(value);
}

function formatFallbackName(refId) {
    if (!refId) return 'Unknown';
    if (typeof refId === 'number') return `ID ${refId}`;
    return refId
        .toString()
        .replace(/_/g, ' ')
        .replace(/\b\w/g, (c) => c.toUpperCase());
}

async function getUserRoleIdentifiers(userId) {
    if (!userId) {
        return {
            roleIds: [],
            roleNames: []
        };
    }

    const [rows] = await db.query(
        `SELECT u.id, u.role_id, r.name as role_name
         FROM users u
         LEFT JOIN roles r ON u.role_id = r.id
         WHERE u.id = ?
         LIMIT 1`,
        [userId]
    );

    if (!rows.length) {
        return {
            roleIds: [],
            roleNames: []
        };
    }

    const roleIds = [];
    const roleNames = [];
    if (rows[0].role_id !== null && rows[0].role_id !== undefined) {
        roleIds.push(normalizeId(rows[0].role_id));
    }
    if (rows[0].role_name) {
        roleNames.push(rows[0].role_name);
    }

    return {
        roleIds,
        roleNames
    };
}

async function resolveAssigneeNames(assignees = [], cache = null) {
    if (!assignees || assignees.length === 0) return [];

    const cacheStore = cache || createAssigneeCache();

    const normalizedAssignees = assignees
        .map(normalizeAssigneeEntry)
        .filter((assignee) => assignee && assignee.refId !== undefined && assignee.refId !== null);

    if (!normalizedAssignees.length) return [];

    const missingUserIds = new Set();
    const missingRoleIds = new Set();
    const missingRoleNames = new Set();

    normalizedAssignees.forEach((assignee) => {
        const refKey = normalizeId(assignee.refId);
        if (assignee.type === 'user') {
            if (!cacheStore.users.has(refKey) && isNumeric(refKey)) {
                missingUserIds.add(refKey);
            }
        } else if (assignee.type === 'role') {
            if (isNumeric(refKey)) {
                if (!cacheStore.rolesById.has(refKey)) {
                    missingRoleIds.add(refKey);
                }
            } else {
                const lower = refKey.toLowerCase();
                if (!cacheStore.rolesByName.has(lower)) {
                    missingRoleNames.add(lower);
                }
            }
        }
    });

    if (missingUserIds.size > 0) {
        const [userRows] = await db.query(
            `SELECT id, name, username, email FROM users WHERE id IN (?)`,
            [[...missingUserIds]]
        );
        userRows.forEach((row) => {
            const key = normalizeId(row.id);
            cacheStore.users.set(
                key,
                row.name || row.username || row.email || `User #${row.id}`
            );
        });
    }

    if (missingRoleIds.size > 0) {
        const [roleRows] = await db.query(
            `SELECT id, name FROM roles WHERE id IN (?)`,
            [[...missingRoleIds]]
        );
        roleRows.forEach((row) => {
            const idKey = normalizeId(row.id);
            const name = row.name || `Role #${row.id}`;
            cacheStore.rolesById.set(idKey, name);
            cacheStore.rolesByName.set(name.toLowerCase(), name);
        });
    }

    if (missingRoleNames.size > 0) {
        const uniqueNames = [...missingRoleNames].filter(
            (name) => !cacheStore.rolesByName.has(name)
        );
        if (uniqueNames.length) {
            const [roleRowsByName] = await db.query(
                `SELECT id, name FROM roles WHERE LOWER(name) IN (?)`,
                [uniqueNames]
            );
            roleRowsByName.forEach((row) => {
                const name = row.name || `Role #${row.id}`;
                cacheStore.rolesById.set(normalizeId(row.id), name);
                cacheStore.rolesByName.set(name.toLowerCase(), name);
            });
        }
    }

    return normalizedAssignees.map((assignee) => {
        const refKey = normalizeId(assignee.refId);
        let displayName;

        if (assignee.type === 'user') {
            displayName = cacheStore.users.get(refKey);
        } else if (assignee.type === 'role') {
            if (isNumeric(refKey)) {
                displayName = cacheStore.rolesById.get(refKey);
            }
            if (!displayName) {
                const lower = refKey.toLowerCase();
                displayName =
                    cacheStore.rolesByName.get(lower) || ROLE_NAME_FALLBACKS[lower];
            }
        }

        return {
            ...assignee,
            name: displayName || formatFallbackName(assignee.refId)
        };
    });
}

function assigneeMatchesUser(assignee, userId, roleIdentifiers = { roleIds: [], roleNames: [] }) {
    if (!assignee || !userId) return false;
    const normalized = normalizeAssigneeEntry(assignee);
    if (!normalized || normalized.refId === undefined || normalized.refId === null) return false;

    const refKey = normalizeId(normalized.refId);

    if (normalized.type === 'user') {
        return refKey === normalizeId(userId);
    }

    if (normalized.type === 'role') {
        if (roleIdentifiers.roleIds.some((roleId) => roleId === refKey)) {
            return true;
        }
        return roleIdentifiers.roleNames.some(
            (roleName) => roleName && roleName.toLowerCase() === refKey.toLowerCase()
        );
    }

    return false;
}

async function getStepApprovalCount(requestId, stepId) {
    const [approvals] = await db.query(
        'SELECT COUNT(*) as count FROM approval_actions WHERE request_id = ? AND step_id = ? AND action = "approve"',
        [requestId, stepId]
    );
    return approvals.length ? approvals[0].count : 0;
}

function parseAssigneeList(rawAssignees) {
    if (!rawAssignees) return [];
    if (Array.isArray(rawAssignees)) {
        return rawAssignees.map(normalizeAssigneeEntry).filter(Boolean);
    }
    if (typeof rawAssignees === 'string') {
        try {
            const parsed = JSON.parse(rawAssignees);
            if (Array.isArray(parsed)) {
                return parsed.map(normalizeAssigneeEntry).filter(Boolean);
            }
        } catch (err) {
            return [];
        }
    }
    return [];
}

function getRequiredApprovals(stepConfig, assigneeCount) {
    if (!stepConfig) return 1;

    if (stepConfig.mode === 'serial') {
        return assigneeCount || 1;
    }

    if (stepConfig.mode === 'parallel' || stepConfig.mode === 'quorum') {
        if (stepConfig.quorum && stepConfig.quorum > 0) {
            return stepConfig.quorum;
        }
        return assigneeCount || 1;
    }

    return 1;
}

function getParallelRequiredApprovals(stepConfig, assigneeCount) {
    if (stepConfig.quorum && stepConfig.quorum > 0) {
        return stepConfig.quorum;
    }
    return assigneeCount || 1;
}

function getSerialNextAssignee(assignees, approvalsCount) {
    if (!assignees.length) return null;
    const currentIndex = Math.min(approvalsCount, assignees.length - 1);
    return assignees[currentIndex];
}

// Helper function for checking user approval permissions
async function checkUserCanApprove(requestId, stepId, userId, roleIdentifiers = null) {
    if (!requestId || !stepId || !userId) return false;

    const [assignment] = await db.query(
        `
        SELECT ws.assignees, ws.mode FROM workflow_steps ws
        JOIN approval_step_progress asp ON ws.id = asp.step_id
        WHERE asp.request_id = ? AND asp.step_id = ? AND asp.status = 'in_progress'
    `,
        [requestId, stepId]
    );

    if (assignment.length === 0) return false;

    const assignees = parseAssigneeList(assignment[0].assignees);
    if (!assignees.length) return false;

    const roleInfo = roleIdentifiers || (await getUserRoleIdentifiers(userId));

    if (assignment[0].mode === 'serial') {
        const approvalsCount = await getStepApprovalCount(requestId, stepId);
        const nextAssignee = getSerialNextAssignee(assignees, approvalsCount);
        if (!nextAssignee) return false;
        return assigneeMatchesUser(nextAssignee, userId, roleInfo);
    }

    return assignees.some((assignee) => assigneeMatchesUser(assignee, userId, roleInfo));
}

// Standalone function for checking approval requirements
function safeParseJson(value, fallback = null) {
    if (!value) return fallback;
    if (typeof value === 'object') return value;
    try {
        return JSON.parse(value);
    } catch (err) {
        return fallback;
    }
}

async function checkApprovalRequiredInternal(module, operation, payload) {
    console.log('🔍 checkApprovalRequiredInternal called with:', { module, operation });

    // Find matching policy
    const [policies] = await db.query(
        `SELECT wp.*, w.name as workflow_name, w.active as workflow_active, w.type as workflow_type
         FROM workflow_policies wp
         JOIN workflows w ON wp.workflow_id = w.id
         WHERE wp.module = ? AND wp.operation = ? AND wp.active = TRUE AND w.active = TRUE
         ORDER BY wp.priority DESC LIMIT 1`,
        [module, operation]
    );

    if (policies.length === 0) {
        return {
            requiresApproval: false,
            requiresNotification: false,
            message: 'No workflow policy found for this operation'
        };
    }

    const policy = policies[0];
    const workflowType = policy.workflow_type || 'approval';
    let requiresApproval = workflowType === 'approval';
    let requiresNotification = workflowType === 'notification';

    // Evaluate condition if present
    const payloadData = safeParseJson(payload, payload);

    if (policy.condition_expr && payloadData) {
        try {
            console.log('Evaluating condition:', policy.condition_expr);
            const conditionResult = evalCondition(policy.condition_expr, payloadData);
            console.log('Condition evaluation result:', conditionResult);

            // For approval workflows, condition determines if approval is required
            // For notification workflows, condition determines if notification is sent
            if (workflowType === 'approval') {
                requiresApproval = conditionResult;
            } else if (workflowType === 'notification') {
                requiresNotification = conditionResult;
            }
        } catch (error) {
            console.warn('Error evaluating condition:', error);
            // Default behavior based on workflow type
        }
    } else {
        console.log(`No condition specified, defaulting to ${workflowType === 'approval' ? 'requiresApproval = true' : 'requiresNotification = true'}`);
    }

    return {
        requiresApproval,
        requiresNotification,
        workflowType,
        workflowId: (requiresApproval || requiresNotification) ? policy.workflow_id : null,
        workflowName: (requiresApproval || requiresNotification) ? policy.workflow_name : null,
        policyId: (requiresApproval || requiresNotification) ? policy.id : null,
        message: requiresApproval ? 'Approval required' :
                requiresNotification ? 'Notification will be sent' : 'No workflow action required'
    };
}

// Simple condition evaluator
function evalCondition(conditionExpr, data) {
    try {
        // Replace variable names with data values
        const expr = conditionExpr.replace(/(\w+)/g, (match) => {
            return data[match] !== undefined ? JSON.stringify(data[match]) : match;
        });
        // Use Function constructor for safer evaluation
        return new Function('data', `return ${expr}`)(data);
    } catch (error) {
        console.error('❌ Condition evaluation error:', error);
        return true; // Default to requiring approval
    }
}

function randomId(prefix) {
    const id = typeof crypto.randomUUID === 'function'
        ? crypto.randomUUID()
        : `${Date.now()}-${Math.random().toString(36).substring(2, 10)}`;
    return prefix ? `${prefix}-${id}` : id;
}

function buildTrackUrl(requestId) {
    if (!requestId) return null;
    const baseUrl = process.env.WORKFLOW_PUBLIC_BASE_URL || '';
    if (!baseUrl) {
        return `/api/workflow/requests/${requestId}/status`;
    }
    return `${baseUrl.replace(/\/$/, '')}/api/workflow/requests/${requestId}/status`;
}

function formatModuleOperationTitle(module, operation) {
    const normalize = (value) => (value || '').replace(/_/g, ' ').replace(/\b\w/g, c => c.toUpperCase());
    return `${normalize(module)} - ${normalize(operation)}`.trim();
}

// Initialize first step for approval request
async function initializeFirstStep(requestId, workflowId) {
    console.log('🔍 initializeFirstStep called with requestId:', requestId, 'workflowId:', workflowId);
    const [firstStep] = await db.query(
        'SELECT * FROM workflow_steps WHERE workflow_id = ? ORDER BY step_order LIMIT 1',
        [workflowId]
    );
    console.log('🔍 firstStep result:', firstStep);

    if (firstStep.length > 0) {
        const step = firstStep[0];
        console.log('🔍 Found step:', step.id);
        const dueAt = step.sla_minutes ?
            new Date(Date.now() + step.sla_minutes * 60000) : null;

        const [insertResult] = await db.query(
            `INSERT INTO approval_step_progress (request_id, step_id, status, due_at)
             VALUES (?, ?, 'in_progress', ?)`,
            [requestId, step.id, dueAt]
        );
        console.log('🔍 Step progress inserted, result:', insertResult);

        // Notify assignees that they need to approve this request
        await notifyStepAssignees(requestId, step);
    } else {
        console.log('❌ No steps found for workflow:', workflowId);
    }
}

// Helper function to notify step assignees
async function notifyStepAssignees(requestId, step) {
    try {
        // Get request details
        const [requests] = await db.query(
            'SELECT title, module, operation FROM approval_requests WHERE id = ?',
            [requestId]
        );

        if (requests.length === 0) return;

        const request = requests[0];
        const assignees = parseAssigneeList(step.assignees);

        // Notify each assignee
        for (const assignee of assignees) {
            if (assignee.type === 'user' && assignee.refId) {
                await notifyWorkflowEvent(assignee.refId, 'approval_needed', {
                    requestId,
                    title: request.title,
                    module: request.module,
                    operation: request.operation,
                    stepId: step.id,
                    stepName: step.name || `Step ${step.step_order}`
                });
            } else if (assignee.type === 'role' && assignee.refId) {
                // For role assignments, we need to find all users with that role
                const [roleUsers] = await db.query(
                    `SELECT u.id FROM users u
                     INNER JOIN user_roles ur ON u.id = ur.user_id
                     WHERE ur.role_id = ?`,
                    [assignee.refId]
                );

                for (const user of roleUsers) {
                    await notifyWorkflowEvent(user.id, 'approval_needed', {
                        requestId,
                        title: request.title,
                        module: request.module,
                        operation: request.operation,
                        stepId: step.id,
                        stepName: step.name || `Step ${step.step_order}`
                    });
                }
            }
        }
    } catch (error) {
        console.error('Failed to notify step assignees:', error);
    }
}

// Audit log function
async function auditLog(eventType, entityType, entityId, actorId, oldValue, newValue, metadata = {}) {
    try {
        await db.query(
            `INSERT INTO workflow_audit_log (event_type, entity_type, entity_id, actor_id, old_value, new_value, metadata)
             VALUES (?, ?, ?, ?, ?, ?, ?)`,
            [
                eventType,
                entityType,
                entityId,
                actorId,
                oldValue ? JSON.stringify(oldValue) : null,
                newValue ? JSON.stringify(newValue) : null,
                JSON.stringify(metadata)
            ]
        );
    } catch (error) {
        console.error('Audit log error:', error);
        // Don't throw - audit logging shouldn't break the main flow
    }
}

class WorkflowController {
    // ========================================
    // HEALTH CHECK
    // ========================================
    async health(req, res) {
        try {
            // DB connectivity
            await db.query('SELECT 1');
            const counters = {};
            const tables = [
                'workflows',
                'workflow_steps',
                'workflow_policies',
                'approval_requests',
                'approval_actions',
                'approval_step_progress',
                'operations',
                'operation_invocations'
            ];
            for (const t of tables) {
                try {
                    const [rows] = await db.query(`SELECT COUNT(*) as count FROM ${t}`);
                    counters[t] = rows[0].count;
                } catch (e) {
                    counters[t] = null; // table missing or not accessible
                }
            }
            const pendingRequests = counters['approval_requests'] === null
                ? null
                : (await db.query(`SELECT COUNT(*) as count FROM approval_requests WHERE status = 'pending'`))[0][0].count;

            let operationMetrics = null;
            try {
                const [invocationStats] = await db.query(
                    `SELECT status, COUNT(*) as count FROM operation_invocations GROUP BY status`
                );
                operationMetrics = invocationStats.reduce((acc, row) => {
                    acc[row.status] = row.count;
                    return acc;
                }, {});
            } catch (metricError) {
                operationMetrics = null;
            }

            return res.json({
                success: true,
                status: 'healthy',
                database: 'connected',
                tables: counters,
                pendingRequests,
                operationMetrics
            });
        } catch (error) {
            return res.status(503).json({
                success: false,
                status: 'unhealthy',
                database: 'disconnected',
                error: error.message
            });
        }
    }
    // ========================================
    // WORKFLOW MANAGEMENT
    // ========================================

    async createWorkflow(req, res) {
        try {
            const { name, module, operation, description, steps } = req.body;
            const active = req.body.active !== undefined ? !!req.body.active : true;
            const userId = req.headers['x-user-id'] || 'system';

            if (!name || !module || !operation || !steps || !Array.isArray(steps)) {
                return res.status(400).json({
                    success: false,
                    message: 'Missing required fields: name, module, operation, steps'
                });
            }

            // Check if workflow already exists for this module and operation
            const [existingWorkflows] = await db.query(
                'SELECT id, name FROM workflows WHERE module = ? AND operation = ? AND active = 1',
                [module, operation]
            );

            if (existingWorkflows.length > 0) {
                return res.status(409).json({
                    success: false,
                    message: `A workflow already exists for ${module} -> ${operation}. You can only edit the existing workflow.`,
                    existingWorkflow: {
                        id: existingWorkflows[0].id,
                        name: existingWorkflows[0].name
                    }
                });
            }

            // Create workflow
            const [workflowResult] = await db.query(
                `INSERT INTO workflows (name, module, operation, description, active, created_by)
                 VALUES (?, ?, ?, ?, ?, ?)`,
                [name, module, operation, description || '', active, userId]
            );

            const workflowId = workflowResult.insertId;

            // Create steps
            for (let i = 0; i < steps.length; i++) {
                const step = steps[i];
                await db.query(
                    `INSERT INTO workflow_steps (workflow_id, step_order, mode, quorum, assignees, sla_minutes)
                     VALUES (?, ?, ?, ?, ?, ?)`,
                    [
                        workflowId,
                        i + 1,
                        step.mode || 'serial',
                        step.quorum || 1,
                        JSON.stringify(step.assignees || []),
                        step.slaMinutes || null
                    ]
                );
            }

            // Log audit
            await auditLog('workflow_created', 'workflow', workflowId, userId, null, { name, module, operation });

            res.status(201).json({
                success: true,
                message: 'Workflow created successfully',
                data: { id: workflowId }
            });
        } catch (error) {
            console.error('Error creating workflow:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to create workflow',
                error: error.message
            });
        }
    }

    async getWorkflows(req, res) {
        try {
            console.log('🔍 getWorkflows called');
            // Test database connection
            const [test] = await db.query('SELECT 1 as test');
            const [dbName] = await db.query('SELECT DATABASE() as db_name');
            console.log('🔍 Database test result:', test, 'DB name:', dbName[0].db_name);

            const { module, operation, active } = req.query;

            let query = `
                SELECT w.*, COUNT(ws.id) as step_count
                FROM workflows w
                LEFT JOIN workflow_steps ws ON w.id = ws.workflow_id
                WHERE 1=1
            `;
            const params = [];

            if (module) {
                query += ' AND w.module = ?';
                params.push(module);
            }

            if (operation) {
                query += ' AND w.operation = ?';
                params.push(operation);
            }

            if (active !== undefined) {
                query += ' AND w.active = ?';
                params.push(active === 'true');
            }

            query += ' GROUP BY w.id ORDER BY w.created_at DESC';

            const [workflows] = await db.query(query, params);

            res.json({
                success: true,
                data: workflows
            });
        } catch (error) {
            console.error('Error fetching workflows:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to fetch workflows'
            });
        }
    }

    async getWorkflowById(req, res) {
        try {
            const { id } = req.params;

            // Get workflow details
            const [workflows] = await db.query(
                'SELECT * FROM workflows WHERE id = ?',
                [id]
            );

            if (workflows.length === 0) {
                return res.status(404).json({
                    success: false,
                    message: 'Workflow not found'
                });
            }

            const workflow = workflows[0];

            // Get workflow steps
                const [steps] = await db.query(
                    'SELECT * FROM workflow_steps WHERE workflow_id = ? ORDER BY step_order',
                [id]
            );

            const assigneeCache = createAssigneeCache();

            // Parse assignees JSON (handle both string and object formats)
            workflow.steps = await Promise.all(
                steps.map(async (step) => {
                    const parsedAssignees = typeof step.assignees === 'string'
                        ? JSON.parse(step.assignees || '[]')
                        : (step.assignees || []);

                    const resolvedAssignees = await resolveAssigneeNames(parsedAssignees, assigneeCache);

                    return {
                    ...step,
                        assignees: resolvedAssignees
                    };
                })
            );

            res.json({
                success: true,
                data: workflow
            });
        } catch (error) {
            console.error('Error fetching workflow by ID:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to fetch workflow'
            });
        }
    }

    async updateWorkflow(req, res) {
        try {
            const { id } = req.params;
            const updateData = req.body;
            const userId = req.headers['x-user-id'] || 'system';

            // Build dynamic update query based on provided fields
            const updateFields = [];
            const updateValues = [];

            if (updateData.name !== undefined) {
                updateFields.push('name = ?');
                updateValues.push(updateData.name);
            }
            if (updateData.module !== undefined) {
                updateFields.push('module = ?');
                updateValues.push(updateData.module);
            }
            if (updateData.operation !== undefined) {
                updateFields.push('operation = ?');
                updateValues.push(updateData.operation);
            }
            if (updateData.description !== undefined) {
                updateFields.push('description = ?');
                updateValues.push(updateData.description);
            }
            if (updateData.active !== undefined) {
                updateFields.push('active = ?');
                updateValues.push(updateData.active ? 1 : 0);
            }

            // Always update the timestamp
            updateFields.push('updated_at = CURRENT_TIMESTAMP');

            if (updateFields.length === 1) {
                // Only timestamp update, nothing to do
                return res.json({
                    success: true,
                    message: 'No changes to update'
                });
            }

            // Update workflow basic info
            const updateQuery = `UPDATE workflows SET ${updateFields.join(', ')} WHERE id = ?`;
            updateValues.push(id);

            await db.query(updateQuery, updateValues);

            // If steps are provided, update them
            if (updateData.steps && Array.isArray(updateData.steps)) {
                // Delete existing steps
                await db.query('DELETE FROM workflow_steps WHERE workflow_id = ?', [id]);

                // Insert new steps
                for (let i = 0; i < updateData.steps.length; i++) {
                    const step = updateData.steps[i];
                    await db.query(
                        `INSERT INTO workflow_steps (workflow_id, step_order, mode, quorum, assignees, sla_minutes)
                         VALUES (?, ?, ?, ?, ?, ?)`,
                        [
                            id,
                            i + 1,
                            step.mode || 'serial',
                            step.quorum || 1,
                            JSON.stringify(step.assignees || []),
                            step.sla_minutes || null
                        ]
                    );
                }
            }

            // Get updated workflow data for audit log
            const [updatedWorkflows] = await db.query('SELECT name, module, operation FROM workflows WHERE id = ?', [id]);
            const updated = updatedWorkflows[0];

            // Log audit
            await auditLog('workflow_updated', 'workflow', id, userId, null, {
                name: updated.name,
                module: updated.module,
                operation: updated.operation
            });

            res.json({
                success: true,
                message: 'Workflow updated successfully'
            });
        } catch (error) {
            console.error('Error updating workflow:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to update workflow',
                error: error.message
            });
        }
    }

    async deleteWorkflow(req, res) {
        try {
            const { id } = req.params;
            const userId = req.headers['x-user-id'] || 'system';

            // Check if workflow exists
            const [workflows] = await db.query('SELECT * FROM workflows WHERE id = ?', [id]);
            if (workflows.length === 0) {
                return res.status(404).json({
                    success: false,
                    message: 'Workflow not found'
                });
            }

            const workflow = workflows[0];

            // Check for dependent records that prevent deletion
            const [policies] = await db.query('SELECT COUNT(*) as count FROM workflow_policies WHERE workflow_id = ?', [id]);
            const [requests] = await db.query('SELECT COUNT(*) as count FROM approval_requests WHERE workflow_id = ?', [id]);

            if (policies[0].count > 0 || requests[0].count > 0) {
                return res.status(409).json({
                    success: false,
                    message: 'Cannot delete workflow with active dependencies',
                    details: {
                        active_policies: policies[0].count,
                        active_requests: requests[0].count
                    },
                    suggestion: 'Deactivate the workflow instead of deleting it, or wait for all approval requests to complete.'
                });
            }

            // Delete workflow steps first (foreign key constraint)
            await db.query('DELETE FROM workflow_steps WHERE workflow_id = ?', [id]);

            // Delete workflow
            await db.query('DELETE FROM workflows WHERE id = ?', [id]);

            // Log audit
            await auditLog('workflow_deleted', 'workflow', id, userId, null, { name: workflow.name, module: workflow.module, operation: workflow.operation });

            res.json({
                success: true,
                message: 'Workflow deleted successfully'
            });
        } catch (error) {
            console.error('Error deleting workflow:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to delete workflow'
            });
        }
    }

    // ========================================
    // POLICY MANAGEMENT
    // ========================================

    async createPolicy(req, res) {
        try {
            const { module, operation, conditionExpr, workflowId, priority = 1 } = req.body;
            const userId = req.headers['x-user-id'] || 'system';

            if (!module || !operation || !workflowId) {
                return res.status(400).json({
                    success: false,
                    message: 'Missing required fields: module, operation, workflowId'
                });
            }

            const [result] = await db.query(
                `INSERT INTO workflow_policies (module, operation, condition_expr, workflow_id, priority, created_by)
                 VALUES (?, ?, ?, ?, ?, ?)`,
                [module, operation, conditionExpr || null, workflowId, priority, userId]
            );

            res.status(201).json({
                success: true,
                message: 'Policy created successfully',
                data: { id: result.insertId }
            });
        } catch (error) {
            console.error('Error creating policy:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to create policy'
            });
        }
    }

    async getPolicies(req, res) {
        try {
            const { module, operation, active = true } = req.query;

            let query = 'SELECT * FROM workflow_policies WHERE 1=1';
            const params = [];

            if (module) {
                query += ' AND module = ?';
                params.push(module);
            }

            if (operation) {
                query += ' AND operation = ?';
                params.push(operation);
            }

            if (active !== undefined) {
                query += ' AND active = ?';
                params.push(active === 'true');
            }

            query += ' ORDER BY priority DESC, created_at DESC';

            const [policies] = await db.query(query, params);

            res.json({
                success: true,
                data: policies
            });
        } catch (error) {
            console.error('Error fetching policies:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to fetch policies'
            });
        }
    }

    // ========================================
    // OPERATIONS INTEGRATION (Module-facing)
    // ========================================

    async registerOperations(req, res) {
        try {
            const serviceName = req.serviceToken?.service || req.body.service;
            const operationsPayload = Array.isArray(req.body.operations)
                ? req.body.operations
                : (req.body.module ? [req.body] : []);

            if (!serviceName) {
                return res.status(400).json({
                    success: false,
                    message: 'Service name is required (token claim or body.service)'
                });
            }

            if (!operationsPayload.length) {
                return res.status(400).json({
                    success: false,
                    message: 'operations array is required'
                });
            }

            const results = [];

            for (const op of operationsPayload) {
                const definition = {
                    service: serviceName,
                    module: op.module,
                    operation: op.operation,
                    method: (op.method || 'POST').toUpperCase(),
                    path: op.path,
                    version: op.version || 'v1',
                    payloadSchema: op.payloadSchema || op.payload_schema || null,
                    isActive: op.isActive !== undefined ? op.isActive : true,
                    createdBy: req.serviceToken?.sub || serviceName
                };

                if (!definition.module || !definition.operation || !definition.path) {
                    return res.status(400).json({
                        success: false,
                        message: 'Each operation requires module, operation, and path'
                    });
                }

                const savedOperation = await operationModel.upsertOperation(definition);
                let callback = null;

                const callbackInput = op.callback || op.callbacks;
                if (callbackInput) {
                    if (!callbackInput.webhook_secret) {
                        return res.status(400).json({
                            success: false,
                            message: 'callback.webhook_secret is required'
                        });
                    }

                    callback = await operationModel.upsertCallback(savedOperation.id, {
                        success_url: callbackInput.successUrl || callbackInput.success_url,
                        failure_url: callbackInput.failureUrl || callbackInput.failure_url,
                        webhook_secret: callbackInput.webhook_secret,
                        retry_policy: callbackInput.retryPolicy || callbackInput.retry_policy
                    });
                }

                await auditLog(
                    'operation_registered',
                    'operation',
                    savedOperation.id,
                    serviceName,
                    null,
                    {
                        method: savedOperation.method,
                        path: savedOperation.path
                    }
                );

                results.push({
                    ...savedOperation,
                    callback: callback
                        ? {
                            success_url: callback.success_url,
                            failure_url: callback.failure_url,
                            retry_policy: callback.retry_policy
                        }
                        : null
                });
            }

            res.status(201).json({
                success: true,
                count: results.length,
                data: results
            });
        } catch (error) {
            console.error('Error registering operations:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to register operations',
                error: error.message
            });
        }
    }

    async listOperations(req, res) {
        try {
            const filters = {
                service: req.query.service,
                module: req.query.module,
                operation: req.query.operation,
                isActive: req.query.active !== undefined ? req.query.active === 'true' : undefined
            };

            let operations;
            try {
                operations = await operationModel.listOperations(filters);
            } catch (error) {
                if (error.code === 'ER_NO_SUCH_TABLE') {
                    console.warn('operations tables missing – running createOperationsRegistryTables...');
                    await createOperationsRegistryTables();
                    operations = await operationModel.listOperations(filters);
                } else {
                    throw error;
                }
            }

            res.json({
                success: true,
                data: operations
            });
        } catch (error) {
            console.error('Error listing operations:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to list operations',
                error: error.message
            });
        }
    }

    async checkOperationPolicy(req, res) {
        try {
            const { module, operation, payload } = req.body;
            if (!module || !operation) {
                return res.status(400).json({
                    success: false,
                    message: 'module and operation are required'
                });
            }

            const result = await checkApprovalRequiredInternal(module, operation, payload);
            res.json({
                success: true,
                data: result
            });
        } catch (error) {
            console.error('Error checking policy:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to evaluate policy'
            });
        }
    }

    async checkApprovalRequired(req, res) {
        try {
            console.log('🔍 checkApprovalRequired called with:', req.query);
            const { module, operation, payload } = req.query;

            if (!module || !operation) {
                return res.status(400).json({
                    success: false,
                    message: 'Missing required parameters: module, operation'
                });
            }

            const result = await checkApprovalRequiredInternal(module, operation, payload);
            res.json(result);
        } catch (error) {
            console.error('Error checking approval requirements:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to check approval requirements'
            });
        }
    }

    async queueOperation(req, res) {
        try {
            const serviceName = req.serviceToken?.service || req.body.service;
            const {
                operationId,
                module,
                operation,
                method,
                path,
                payload,
                requesterId,
                requesterName,
                resourceRef,
                correlationId,
                idempotencyKey: bodyIdempotencyKey
            } = req.body;

            if (!payload) {
                return res.status(400).json({
                    success: false,
                    message: 'payload is required'
                });
            }

            // Normalize path by removing trailing slash
            let normalizedPath = path;
            if (normalizedPath && normalizedPath.length > 1 && normalizedPath.endsWith('/')) {
                normalizedPath = normalizedPath.slice(0, -1);
            }

            let operationRecord = null;
            if (operationId) {
                operationRecord = await operationModel.getOperationById(operationId);
            } else if (serviceName && module && operation && method && normalizedPath) {
                operationRecord = await operationModel.getOperationBySignature({
                    service: serviceName,
                    module,
                    operation,
                    method: (method || 'POST').toUpperCase(),
                    path: normalizedPath
                });
            } else if (serviceName && module && operation) {
                const candidates = await operationModel.listOperations({
                    service: serviceName,
                    module,
                    operation,
                    isActive: true
                });
                operationRecord = candidates[0];
            }

            if (!operationRecord) {
                return res.status(404).json({
                    success: false,
                    message: 'Operation not registered for workflow orchestration'
                });
            }

            const normalizedModule = module || operationRecord.module;
            const normalizedOperation = operation || operationRecord.operation;
            const normalizedPayload = typeof payload === 'string' ? safeParseJson(payload, payload) : payload;

            const idempotencyKey = req.headers['idempotency-key'] || bodyIdempotencyKey || null;
            if (idempotencyKey) {
                const existingInvocation = await operationModel.findInvocationByIdempotency(operationRecord.id, idempotencyKey);
                if (existingInvocation) {
                    const response = {
                    success: true,
                        requiresApproval: existingInvocation.status === 'queued',
                        data: {
                            invocationId: existingInvocation.id,
                            operationId: operationRecord.id,
                            approvalRequestId: existingInvocation.approval_request_id,
                            status: existingInvocation.status,
                            trackUrl: buildTrackUrl(existingInvocation.approval_request_id)
                        }
                    };
                    return res.status(existingInvocation.status === 'queued' ? 202 : 200).json(response);
                }
            }

            const checkResult = await checkApprovalRequiredInternal(
                normalizedModule,
                normalizedOperation,
                normalizedPayload
            );

            if (checkResult.requiresApproval && !requesterId) {
                return res.status(400).json({
                    success: false,
                    message: 'requesterId is required when approval is needed'
                });
            }

            const correlation = correlationId || randomId('corr');
            const resource = resourceRef || randomId('res');
            const createdBy = req.serviceToken?.sub || serviceName || 'system-service';

            let approvalRequestId = null;

            if (checkResult.requiresApproval) {
                const title = formatModuleOperationTitle(normalizedModule, normalizedOperation);
            const [requestResult] = await db.query(
                `INSERT INTO approval_requests
                     (workflow_id, title, module, operation, request_type, requester_id, requester_name, payload, resource_ref, correlation_id, status, current_step_order)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                [
                    checkResult.workflowId,
                    title,
                        normalizedModule,
                        normalizedOperation,
                        'approval', // request_type
                    requesterId,
                    requesterName || requesterId,
                        JSON.stringify(normalizedPayload || {}),
                        resource,
                        correlation,
                        'pending', // status
                        1 // current_step_order
                    ]
                );

                approvalRequestId = requestResult.insertId;

                // Log system action: workflow request created
                await logSystemAction(approvalRequestId, null, 'request_created', 'System', `Approval request created for ${normalizedModule} - ${normalizedOperation}`);

                await initializeFirstStep(approvalRequestId, checkResult.workflowId);

                // Emit WebSocket event for real-time updates
                emitToRoom('approvals', 'new_approval_request', {
                    requestId: approvalRequestId,
                    workflowId: checkResult.workflowId,
                    title,
                    module: normalizedModule,
                    operation: normalizedOperation,
                    requesterId,
                    requesterName: requesterName || requesterId,
                    resourceRef: resource,
                    correlationId: correlation,
                    timestamp: new Date().toISOString()
                });

                // Create notification for request creator (if different from requester)
                if (createdBy && createdBy !== requesterId && createdBy !== 'system-service') {
                    await notifyWorkflowEvent(createdBy, 'request_created', {
                        requestId: approvalRequestId,
                        title,
                        module: normalizedModule,
                        operation: normalizedOperation
                    });
                }

                await auditLog('request_created', 'request', approvalRequestId, requesterId || createdBy, null, {
                    module: normalizedModule,
                    operation: normalizedOperation,
                    workflowId: checkResult.workflowId
                });
            }

            const invocation = await operationModel.recordInvocation({
                operationId: operationRecord.id,
                approvalRequestId,
                correlationId: correlation,
                resourceRef: resource,
                payload: normalizedPayload || {},
                createdBy,
                status: checkResult.requiresApproval ? 'queued' : 'approved',
                idempotencyKey
            });

            await auditLog('operation_invoked', 'operation_invocation', invocation.id, createdBy, null, {
                operationId: operationRecord.id,
                requiresApproval: checkResult.requiresApproval,
                requestId: approvalRequestId
            });

            if (!checkResult.requiresApproval) {
                return res.status(200).json({
                success: true,
                    requiresApproval: false,
                data: {
                        invocationId: invocation.id,
                        operationId: operationRecord.id,
                        status: 'approved',
                        resourceRef: resource,
                        correlationId: correlation
                    }
                });
            }

            return res.status(202).json({
                success: true,
                    requiresApproval: true,
                data: {
                    requestId: approvalRequestId,
                    invocationId: invocation.id,
                    workflowId: checkResult.workflowId,
                    status: 'pending',
                    trackUrl: buildTrackUrl(approvalRequestId),
                    resourceRef: resource,
                    correlationId: correlation
                }
            });
        } catch (error) {
            console.error('❌ Error queuing operation:', error);
            console.error('❌ Stack trace:', error.stack);
            res.status(500).json({
                success: false,
                message: 'Failed to queue operation',
                error: error.message
            });
        }
    }

    async queueOperationTest(req, res) {
        try {
            const userId = req.headers['x-user-id'] || req.user?.id || 'workflow-test';
            const {
                module,
                operation,
                payload,
                resourceRef,
                correlationId,
                service = 'supply_chain_frontend',
                requesterId = userId,
                requesterName = req.headers['x-user-name'] || req.user?.name || 'Workflow Tester'
            } = req.body;

            console.log('🔄 QueueOperationTest called with:', { module, operation, requesterId });

            // Validate required fields
            if (!module || !operation || !payload) {
                return res.status(400).json({
                    success: false,
                    message: 'Missing required fields: module, operation, payload'
                });
            }

            // Check if approval or notification is required
            const approvalCheck = await checkApprovalRequiredInternal(module, operation, payload);
            if (!approvalCheck.requiresApproval && !approvalCheck.requiresNotification) {
                return res.status(400).json({
                    success: false,
                    message: 'Neither approval nor notification required for this operation'
                });
            }

            // Handle notification workflows (send notification only)
            if (approvalCheck.workflowType === 'notification') {
                console.log('📢 Sending notification for notification-type workflow');

                // Get workflow details for notification
                const [workflowRows] = await db.query('SELECT * FROM workflows WHERE id = ?', [approvalCheck.workflowId]);
                const workflow = workflowRows[0];

                // Send notification
                await notifyWorkflowEvent(requesterId, 'request_created', {
                    workflow: workflow,
                    requester: { id: requesterId, name: requesterName },
                    payload: payload,
                    message: `${module} - ${operation}: Notification sent (no approval required)`
                });

                return res.json({
                    success: true,
                    message: 'Notification sent successfully',
                    workflowType: 'notification',
                    workflowId: approvalCheck.workflowId
                });
            }

            // Handle approval workflows (create approval request)
            const workflowId = approvalCheck.workflowId;
            const requestData = {
                workflow_id: workflowId,
                module: module,
                operation: operation,
                requester_id: requesterId,
                requester_name: requesterName,
                payload: JSON.stringify(payload),
                status: 'pending',
                resource_ref: resourceRef,
                correlation_id: correlationId,
                created_at: new Date(),
                updated_at: new Date()
            };

            const [result] = await db.query(
                `INSERT INTO approval_requests
                 (workflow_id, module, operation, requester_id, requester_name, payload, status, resource_ref, correlation_id, created_at, updated_at)
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                [
                    requestData.workflow_id,
                    requestData.module,
                    requestData.operation,
                    requestData.requester_id,
                    requestData.requester_name,
                    requestData.payload,
                    requestData.status,
                    requestData.resource_ref,
                    requestData.correlation_id,
                    requestData.created_at,
                    requestData.updated_at
                ]
            );

            const requestId = result.insertId;

            // Create initial timeline entry
            await logSystemAction(requestId, null, 'request_created', requesterName, 'Request submitted for approval');

            // Get workflow details for response
            const [workflowRows] = await db.query('SELECT * FROM workflows WHERE id = ?', [workflowId]);
            const workflow = workflowRows[0];

            // Notify relevant users
            await notifyWorkflowEvent(requestId, 'request_created', {
                workflow: workflow,
                requester: { id: requesterId, name: requesterName },
                payload: payload
            });

            return res.status(201).json({
                success: true,
                message: 'Operation queued for approval successfully',
                data: {
                    requestId: requestId,
                    workflowId: workflowId,
                    workflowName: workflow.name,
                    status: 'pending'
                }
            });

        } catch (error) {
            console.error('❌ Error running workflow test:', error);
            return res.status(500).json({
                success: false,
                message: 'Workflow test run failed',
                error: error.message
            });
        }
    }

    // ========================================
    // APPROVAL ACTIONS (User-facing)
    // ========================================

    async getRequests(req, res) {
        const execute = async () => {
            const { assignee, module, operation, status, actor } = req.query;
            const userId = req.headers['x-user-id'] || req.user?.id;

            if (!assignee && !actor && !userId) {
                return res.status(400).json({
                    success: false,
                    message: 'x-user-id header is required for request filtering'
                });
            }

            let query = `
                SELECT ar.*, w.name as workflow_name,
                       asp.step_id, asp.status as step_status, asp.due_at
                FROM approval_requests ar
                JOIN workflows w ON ar.workflow_id = w.id
                LEFT JOIN approval_step_progress asp ON ar.id = asp.request_id AND asp.status = 'in_progress'
                WHERE 1=1
            `;
            const params = [];

            if (assignee) {
                query += ` AND EXISTS (
                    SELECT 1 FROM workflow_steps ws
                    JOIN approval_step_progress asp2 ON ws.id = asp2.step_id
                    WHERE asp2.request_id = ar.id AND asp2.status = 'in_progress'
                    AND JSON_CONTAINS(ws.assignees, JSON_OBJECT('refId', ?))
                )`;
                params.push(assignee);
            }

            if (module) {
                query += ' AND ar.module = ?';
                params.push(module);
            }

            if (operation) {
                query += ' AND ar.operation = ?';
                params.push(operation);
            }

            if (status) {
                query += ' AND ar.status = ?';
                params.push(status);
            }

            if (actor) {
                query += ` AND EXISTS (
                    SELECT 1 FROM approval_actions aa
                    WHERE aa.request_id = ar.id AND aa.actor_id = ?
                )`;
                params.push(actor);
            }

            query += ' ORDER BY ar.created_at DESC';

            let [requests] = await db.query(query, params);

            if (!assignee && !actor && userId) {
                const roleIdentifiers = await getUserRoleIdentifiers(userId);
                const matches = await Promise.all(
                    requests.map((request) => {
                        if (!request.step_id) return Promise.resolve(false);
                        return checkUserCanApprove(
                            request.id,
                            request.step_id,
                            userId,
                            roleIdentifiers
                        );
                    })
                );
                requests = requests.filter((_, idx) => matches[idx]);
            }

            res.json({
                success: true,
                data: requests
            });
        };

        try {
            await execute();
        } catch (error) {
            const needsMigration = error.code === 'ER_BAD_FIELD_ERROR'
                || (typeof error.sqlMessage === 'string' && error.sqlMessage.includes('workflow_id'));

            if (error.code === 'ER_NO_SUCH_TABLE' || needsMigration) {
                try {
                    if (error.code === 'ER_NO_SUCH_TABLE') {
                        console.warn('Approval tables missing; creating central workflow tables...');
                        await createCentralWorkflowTables();
                    } else {
                        console.warn('Approval tables outdated; running migration to add payload columns...');
                        await migrateApprovalTablesForPayloads();
                    }
                    await execute();
                    return;
                } catch (retryError) {
                    console.error('Retry after creating/migrating tables failed:', retryError);
                    return res.status(500).json({
                        success: false,
                        message: 'Failed to fetch requests after repairing tables',
                        error: retryError.message
                    });
                }
            }

            console.error('Error fetching requests:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to fetch requests',
                error: error.message
            });
        }
    }

    async getAllPayloadsWithLogs(req, res) {
        try {
            const {
                module,
                status,
                userId,
                dateFrom,
                dateTo,
                limit = 50,
                offset = 0,
                search
            } = req.query;

            // Base query for approval requests with aggregated data
            let query = `
                SELECT
                    ar.id,
                    ar.title,
                    ar.module,
                    ar.operation,
                    ar.status,
                    ar.requester_id,
                    ar.requester_name,
                    ar.payload,
                    ar.created_at,
                    ar.updated_at,
                    ar.workflow_id,
                    w.name as workflow_name,
                    ar.current_step_order,
                    COUNT(DISTINCT aa.id) as total_actions,
                    COUNT(DISTINCT CASE WHEN aa.action = 'approve' THEN aa.id END) as approvals_gotten,
                    COUNT(DISTINCT wd.id) as webhook_attempts,
                    COUNT(DISTINCT CASE WHEN wd.success = 1 THEN wd.id END) as webhook_successes,
                    MAX(aa.created_at) as last_action_at
                FROM approval_requests ar
                LEFT JOIN workflows w ON ar.workflow_id = w.id
                LEFT JOIN approval_actions aa ON ar.id = aa.request_id
                LEFT JOIN webhook_deliveries wd ON ar.id = wd.request_id
                WHERE 1=1
            `;

            const params = [];

            // Add filters
            if (module) {
                query += ' AND ar.module = ?';
                params.push(module);
            }

            if (status && status !== 'all') {
                query += ' AND ar.status = ?';
                params.push(status);
            }

            if (userId) {
                query += ' AND ar.requester_id = ?';
                params.push(userId);
            }

            if (dateFrom) {
                query += ' AND ar.created_at >= ?';
                params.push(dateFrom);
            }

            if (dateTo) {
                query += ' AND ar.created_at <= ?';
                params.push(dateTo);
            }

            if (search) {
                query += ' AND (ar.title LIKE ? OR ar.operation LIKE ?)';
                params.push(`%${search}%`, `%${search}%`);
            }

            // Group by and add pagination
            query += ' GROUP BY ar.id ORDER BY ar.created_at DESC LIMIT ? OFFSET ?';
            params.push(parseInt(limit), parseInt(offset));

            const [payloads] = await db.query(query, params);

            // Get total count for pagination
            const countQuery = `
                SELECT COUNT(DISTINCT ar.id) as total
                FROM approval_requests ar
                WHERE 1=1
            `;
            const countParams = [];

            if (module) {
                countQuery += ' AND ar.module = ?';
                countParams.push(module);
            }
            if (status && status !== 'all') {
                countQuery += ' AND ar.status = ?';
                countParams.push(status);
            }
            if (userId) {
                countQuery += ' AND ar.requester_id = ?';
                countParams.push(userId);
            }
            if (dateFrom) {
                countQuery += ' AND ar.created_at >= ?';
                countParams.push(dateFrom);
            }
            if (dateTo) {
                countQuery += ' AND ar.created_at <= ?';
                countParams.push(dateTo);
            }
            if (search) {
                countQuery += ' AND (ar.title LIKE ? OR ar.operation LIKE ?)';
                countParams.push(`%${search}%`, `%${search}%`);
            }

            const [countResult] = await db.query(countQuery, countParams);
            const total = countResult[0]?.total || 0;

            // Categorize payloads and enrich with workflow step data
            for (const payload of payloads) {
                // Parse payload if needed
                payload.payload = safeParseJson(payload.payload, payload.payload);

                // Get workflow steps to calculate required approvals
                const [steps] = await db.query(
                    'SELECT * FROM workflow_steps WHERE workflow_id = ? ORDER BY step_order',
                    [payload.workflow_id]
                );

                let requiredApprovals = 0;
                if (steps.length > 0) {
                    // For simplicity, use the current step's requirements
                    const currentStep = steps.find(s => s.step_order === payload.current_step_order) || steps[0];
                    if (currentStep) {
                        const assignees = parseAssigneeList(currentStep.assignees || []);
                        requiredApprovals = getRequiredApprovals(currentStep, assignees.length);
                    }
                }

                payload.required_approvals = requiredApprovals;

                // Determine status category
                payload.status_category = this.categorizePayloadStatus(payload);

                // Add execution status
                payload.execution_status = payload.webhook_successes > 0 ? 'executed' :
                                         payload.status === 'approved' ? 'pending_execution' :
                                         'not_ready';
            }

            res.json({
                success: true,
                data: payloads,
                pagination: {
                    total,
                    limit: parseInt(limit),
                    offset: parseInt(offset),
                    hasMore: offset + payloads.length < total
                }
            });

        } catch (error) {
            console.error('Error fetching payloads with logs:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to fetch payloads with logs',
                error: error.message
            });
        }
    }

    categorizePayloadStatus(payload) {
        const { status, current_step_order, approvals_gotten, required_approvals } = payload;

        if (status === 'pending') {
            if (current_step_order === 1 && approvals_gotten === 0) {
                return 'pending';
            } else if (approvals_gotten > 0 && approvals_gotten < required_approvals) {
                return 'waiting';
            }
        } else if (status === 'approved') {
            return 'approved';
        } else if (status === 'changes_required' || status === 'cancelled') {
            return 'rejected';
        }

        return 'unknown';
    }

    async getWebhookDeliveries(req, res) {
        try {
            const { id } = req.params;

            // Fetch webhook deliveries for this request
            const [deliveries] = await db.query(
                'SELECT * FROM webhook_deliveries WHERE request_id = ? ORDER BY created_at DESC',
                [id]
            );

            // Parse payloads for display
            for (const delivery of deliveries) {
                try {
                    delivery.payload = JSON.parse(delivery.payload);
                } catch (e) {
                    // If parsing fails, keep as string
                    delivery.payload = delivery.payload;
                }
            }

            res.json({
                success: true,
                data: deliveries
            });

        } catch (error) {
            console.error('Error fetching webhook deliveries:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to fetch webhook deliveries',
                error: error.message
            });
        }
    }

    async getApprovalActions(req, res) {
        try {
            const { id } = req.params;

            // Fetch approval actions for this request
            const [actions] = await db.query(`
                SELECT
                    aa.id,
                    aa.request_id,
                    aa.step_id,
                    aa.actor_id,
                    aa.actor_name,
                    aa.action,
                    aa.comment,
                    aa.created_at,
                    CONCAT('Step ', ws.step_order) as step_name,
                    ws.step_order,
                    ws.assignees,
                    ws.mode,
                    ws.quorum,
                    -- Calculate elapsed seconds from request creation
                    TIMESTAMPDIFF(SECOND, ar.created_at, aa.created_at) as elapsed_seconds
                FROM approval_actions aa
                LEFT JOIN workflow_steps ws ON aa.step_id = ws.id
                LEFT JOIN approval_requests ar ON aa.request_id = ar.id
                WHERE aa.request_id = ?
                ORDER BY aa.created_at ASC
            `, [id]);

            // Get workflow steps with their assignees for this request
            const [request] = await db.query(`
                SELECT workflow_id FROM approval_requests WHERE id = ?
            `, [id]);

            if (request.length === 0) {
                return res.json({ success: true, data: [] });
            }

            const [steps] = await db.query(`
                SELECT 
                    ws.id,
                    ws.step_order,
                    ws.assignees,
                    ws.mode,
                    ws.quorum,
                    CONCAT('Step ', ws.step_order) as step_name
                FROM workflow_steps ws
                WHERE ws.workflow_id = ?
                ORDER BY ws.step_order ASC
            `, [request[0].workflow_id]);

            // Get all approvals grouped by step
            const [stepApprovals] = await db.query(`
                SELECT 
                    aa.step_id,
                    aa.actor_id,
                    aa.actor_name,
                    aa.action,
                    aa.created_at
                FROM approval_actions aa
                WHERE aa.request_id = ? 
                AND aa.action IN ('approve', 'reject')
                ORDER BY aa.created_at ASC
            `, [id]);

            // Build enhanced approval data with assignee information
            const enhancedActions = actions.map(action => {
                const actionData = { ...action };
                
                // Find the corresponding step
                const step = steps.find(s => s.id === action.step_id);
                if (step && step.assignees) {
                    try {
                        actionData.step_assignees = typeof step.assignees === 'string' 
                            ? JSON.parse(step.assignees) 
                            : step.assignees;
                        actionData.step_mode = step.mode;
                        actionData.step_quorum = step.quorum;
                        
                        // Get approvals for this step
                        const stepApprovalList = stepApprovals.filter(sa => sa.step_id === action.step_id);
                        
                        // Match approvals with assignees
                        actionData.assignees_status = actionData.step_assignees.map(assignee => {
                            const approval = stepApprovalList.find(sa => 
                                sa.actor_id == assignee.refId || 
                                sa.actor_name.toLowerCase().includes(assignee.name.toLowerCase())
                            );
                            
                            return {
                                type: assignee.type,
                                refId: assignee.refId,
                                name: assignee.name,
                                hasApproved: !!approval,
                                action: approval ? approval.action : null,
                                approvedAt: approval ? approval.created_at : null,
                                approverName: approval ? approval.actor_name : null
                            };
                        });
                    } catch (e) {
                        console.error('Error parsing assignees:', e);
                    }
                }
                
                return actionData;
            });

            res.json({
                success: true,
                data: enhancedActions,
                steps: steps.map(step => ({
                    ...step,
                    assignees: typeof step.assignees === 'string' 
                        ? JSON.parse(step.assignees) 
                        : step.assignees
                }))
            });

        } catch (error) {
            console.error('Error fetching approval actions:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to fetch approval actions',
                error: error.message
            });
        }
    }

    async getRequestById(req, res) {
        try {
            const { id } = req.params;

            const [requests] = await db.query(
                `SELECT ar.*, w.name as workflow_name
                 FROM approval_requests ar
                 JOIN workflows w ON ar.workflow_id = w.id
                 WHERE ar.id = ?`,
                [id]
            );

            if (requests.length === 0) {
                return res.status(404).json({
                    success: false,
                    message: 'Request not found'
                });
            }

            const request = requests[0];
            request.payload = safeParseJson(request.payload, request.payload);

            const [steps] = await db.query(
                `SELECT ws.*, asp.status as progress_status, asp.started_at, asp.completed_at, asp.due_at
                 FROM workflow_steps ws
                 LEFT JOIN approval_step_progress asp ON ws.id = asp.step_id AND asp.request_id = ?
                 WHERE ws.workflow_id = ?
                 ORDER BY ws.step_order`,
                [id, request.workflow_id]
            );

            const [actions] = await db.query(
                `SELECT aa.*, u.name as actor_full_name
                 FROM approval_actions aa
                 LEFT JOIN users u ON aa.actor_id = u.id
                 WHERE aa.request_id = ?
                 ORDER BY aa.created_at ASC`,
                [id]
            );

            const stepNameMap = new Map();
            steps.forEach(step => stepNameMap.set(step.id, step.name || `Step ${step.step_order}`));

            const timeline = actions.map(action => {
                const createdAt = action.created_at ? new Date(action.created_at) : null;
                const requestCreated = request.created_at ? new Date(request.created_at) : null;
                const elapsedMs = createdAt && requestCreated ? (createdAt - requestCreated) : null;

                return {
                    id: action.id,
                    step_id: action.step_id,
                    step_name: stepNameMap.get(action.step_id) || 'Step',
                    actor_id: action.actor_id,
                    actor_name: action.actor_name || action.actor_full_name || `User #${action.actor_id}`,
                    action: action.action,
                    comment: action.comment,
                    created_at: action.created_at,
                    elapsed_seconds: elapsedMs !== null ? Math.max(Math.floor(elapsedMs / 1000), 0) : null
                };
            });

            for (const step of steps) {
                const assignees = parseAssigneeList(step.assignees);
                step.assignees = assignees;

                const stepHistory = timeline.filter(
                    entry => entry.step_id === step.id && entry.action === 'approve'
                );

                const assigneeCount = assignees.length || 1;
                const requiredApprovals = getRequiredApprovals(step, assigneeCount);
                const approvalsCount = stepHistory.length;

                step.actions = stepHistory;
                step.progress_summary = {
                    total_assignees: assigneeCount,
                    required_approvals: requiredApprovals,
                    approvals_completed: approvalsCount,
                    approvals_remaining: Math.max(requiredApprovals - approvalsCount, 0)
                };
            }

            request.steps = steps;
            request.history = timeline;

            res.json({
                success: true,
                data: request
            });
        } catch (error) {
            console.error('Error fetching request:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to fetch request'
            });
        }
    }

    async getRequestStatus(req, res) {
        try {
            const { id } = req.params;
            const [requests] = await db.query(
                `SELECT id, workflow_id, status, module, operation, resource_ref, correlation_id, payload, created_at, updated_at
                 FROM approval_requests WHERE id = ?`,
                [id]
            );

            if (!requests.length) {
                return res.status(404).json({
                    success: false,
                    message: 'Request not found'
                });
            }

            const request = requests[0];
            request.payload = safeParseJson(request.payload, null);

            const invocation = await operationModel.getInvocationByRequestId(id);

            res.json({
                success: true,
                data: {
                    request,
                    invocation
                }
            });
        } catch (error) {
            console.error('Error fetching request status:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to fetch request status'
            });
        }
    }

    async approveRequest(req, res) {
        try {
            const { id } = req.params;
            const { stepId, comment } = req.body;
            const userId = req.headers['x-user-id'];
            const userName = req.headers['x-user-name'] || userId;

            if (!userId) {
                return res.status(400).json({
                    success: false,
                    message: 'Missing approver user ID'
                });
            }

            // Check if user can approve this step
            const roleIdentifiers = await getUserRoleIdentifiers(userId);
            const canApprove = await checkUserCanApprove(id, stepId, userId, roleIdentifiers);
            if (!canApprove) {
                return res.status(403).json({
                    success: false,
                    message: 'User not authorized to approve this step'
                });
            }

            // Record approval action
            await db.query(
                `INSERT INTO approval_actions (request_id, step_id, actor_id, actor_name, action, comment)
                 VALUES (?, ?, ?, ?, 'approve', ?)`,
                [id, stepId, userId, userName, comment || '']
            );

            // Update step progress and check if step is complete
            const stepComplete = await this.updateStepProgress(id, stepId, 'approve');

            if (stepComplete) {
                // Check if workflow is complete
                const workflowComplete = await this.checkWorkflowComplete(id);
                if (workflowComplete) {
                    await this.finalizeApprovedRequest(id);
                } else {
                    await this.advanceToNextStep(id);
                }
            }

            // Log audit
            await auditLog('request_approved', 'request', id, userId, null, { stepId, comment });

            // Emit WebSocket event for real-time updates
            emitToRoom('approvals', 'request_approved', {
                requestId: id,
                stepId,
                approverId: userId,
                approverName: userName,
                comment,
                timestamp: new Date().toISOString()
            });

            // Get request details for notification
            const request = await this.fetchRequestRecord(id);
            if (request) {
                // Notify requester that their request was approved
                if (request.requester_id && request.requester_id !== userId) {
                    await notifyWorkflowEvent(request.requester_id, 'request_approved', {
                        requestId: id,
                        title: request.title,
                        module: request.module,
                        operation: request.operation,
                        approverName: userName,
                        comment
                    });
                }
            }

            res.json({
                success: true,
                message: 'Request approved successfully'
            });
        } catch (error) {
            console.error('Error approving request:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to approve request'
            });
        }
    }

    async rejectRequest(req, res) {
        try {
            const { id } = req.params;
            const { stepId, comment } = req.body;
            const userId = req.headers['x-user-id'];
            const userName = req.headers['x-user-name'] || userId;

            if (!userId) {
                return res.status(400).json({
                    success: false,
                    message: 'Missing approver user ID'
                });
            }

            // Check authorization (same as approve)
            const roleIdentifiers = await getUserRoleIdentifiers(userId);
            const canApprove = await checkUserCanApprove(id, stepId, userId, roleIdentifiers);
            if (!canApprove) {
                return res.status(403).json({
                    success: false,
                    message: 'User not authorized to reject this step'
                });
            }

            // Record rejection action
            await db.query(
                `INSERT INTO approval_actions (request_id, step_id, actor_id, actor_name, action, comment)
                 VALUES (?, ?, ?, ?, 'reject', ?)`,
                [id, stepId, userId, userName, comment || '']
            );

            // Mark request as changes_required (no branching)
            await db.query(
                'UPDATE approval_requests SET status = ?, updated_at = NOW() WHERE id = ?',
                ['changes_required', id]
            );

            // Log system action: workflow rejected
            await logSystemAction(id, stepId, 'workflow_rejected', 'System', `Workflow rejected: ${comment || 'No reason provided'}`);

            const updatedRequest = await this.fetchRequestRecord(id);
            if (updatedRequest) {
                await this.notifyOperationCallback(updatedRequest, 'rejected');
            }

            // Log audit
            await auditLog('request_rejected', 'request', id, userId, null, { stepId, comment });

            // Emit WebSocket event for real-time updates
            emitToRoom('approvals', 'request_rejected', {
                requestId: id,
                stepId,
                rejectorId: userId,
                rejectorName: userName,
                comment,
                timestamp: new Date().toISOString()
            });

            // Get request details for notification
            const request = await this.fetchRequestRecord(id);
            if (request) {
                // Notify requester that their request was rejected
                if (request.requester_id && request.requester_id !== userId) {
                    await notifyWorkflowEvent(request.requester_id, 'request_rejected', {
                        requestId: id,
                        title: request.title,
                        module: request.module,
                        operation: request.operation,
                        rejectorName: userName,
                        comment
                    });
                }
            }

            res.json({
                success: true,
                message: 'Request marked as changes required'
            });
        } catch (error) {
            console.error('Error rejecting request:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to reject request'
            });
        }
    }

    // ========================================
    // HELPER METHODS
    // ========================================



    updateStepProgress = async (requestId, stepId, action) => {
        const [step] = await db.query(
            'SELECT * FROM workflow_steps WHERE id = ?',
            [stepId]
        );

        if (step.length === 0) return false;

        const stepConfig = step[0];
        const assignees = parseAssigneeList(stepConfig.assignees);
        const assigneeCount = assignees.length || 1;
        const approvalsCount = await getStepApprovalCount(requestId, stepId);

        let stepComplete = false;

        if (stepConfig.mode === 'serial') {
            stepComplete = approvalsCount >= assigneeCount;
        } else if (stepConfig.mode === 'parallel') {
            const requiredApprovals = getParallelRequiredApprovals(stepConfig, assigneeCount);
            stepComplete = approvalsCount >= requiredApprovals;
        } else if (stepConfig.mode === 'quorum') {
            const requiredQuorum = stepConfig.quorum || 1;
            stepComplete = approvalsCount >= requiredQuorum;
        } else {
            stepComplete = approvalsCount >= 1;
        }

        if (stepComplete) {
            await db.query(
                'UPDATE approval_step_progress SET status = ?, completed_at = NOW() WHERE request_id = ? AND step_id = ?',
                ['completed', requestId, stepId]
            );

            // Log system action: step completed
            await logSystemAction(requestId, stepId, 'step_completed', 'System', `Step ${stepConfig.step_order} completed with ${approvalsCount} approval(s)`);
        }

        return stepComplete;
    }

    checkWorkflowComplete = async (requestId) => {
        const [pendingSteps] = await db.query(
            'SELECT COUNT(*) as count FROM approval_step_progress WHERE request_id = ? AND status != "completed"',
            [requestId]
        );

        return pendingSteps[0].count === 0;
    }

    advanceToNextStep = async (requestId) => {
        // Get current step order
        const [currentRequest] = await db.query(
            'SELECT current_step_order FROM approval_requests WHERE id = ?',
            [requestId]
        );

        if (currentRequest.length === 0) return;

        const nextOrder = currentRequest[0].current_step_order + 1;

        // Check if next step exists
        const [nextStep] = await db.query(
            `SELECT ws.* FROM workflow_steps ws
             JOIN approval_requests ar ON ws.workflow_id = ar.workflow_id
             WHERE ar.id = ? AND ws.step_order = ?`,
            [requestId, nextOrder]
        );

        if (nextStep.length > 0) {
            const step = nextStep[0];
            const dueAt = step.sla_minutes ?
                new Date(Date.now() + step.sla_minutes * 60000) : null;

            await db.query(
                `INSERT INTO approval_step_progress (request_id, step_id, status, due_at)
                 VALUES (?, ?, 'in_progress', ?)`,
                [requestId, step.id, dueAt]
            );

            await db.query(
                'UPDATE approval_requests SET current_step_order = ?, updated_at = NOW() WHERE id = ?',
                [nextOrder, requestId]
            );

            // Log system action: step started
            await logSystemAction(requestId, step.id, 'step_started', 'System', `Step ${nextOrder} assigned to approvers`);

            // Notify assignees of the new step that it's their turn to approve
            await notifyStepAssignees(requestId, step);
        }
    }

    finalizeApprovedRequest = async (requestId) => {
        // Update request status
        await db.query(
            'UPDATE approval_requests SET status = ?, updated_at = NOW() WHERE id = ?',
            ['approved', requestId]
        );

        // Log system action: workflow completed
        await logSystemAction(requestId, null, 'workflow_completed', 'System', 'Workflow approved and completed');

        const request = await this.fetchRequestRecord(requestId);
        if (request) {
            await this.notifyOperationCallback(request, 'approved');

            // Notify requester that workflow is complete
            if (request.requester_id) {
                await notifyWorkflowEvent(request.requester_id, 'workflow_completed', {
                    requestId,
                    title: request.title,
                    module: request.module,
                    operation: request.operation
                });
            }
        }
    }

    executeCallback = async ({ url, data, requestId, secret }) => {
        console.log(`📡 EXECUTING CALLBACK: ${url} for request ${requestId}`);
        const headers = {
            'Content-Type': 'application/json',
            'x-service-token': process.env.SERVICE_TOKEN || 'internal-service'
        };

        if (secret) {
            const signature = crypto.createHmac('sha256', secret).update(JSON.stringify(data)).digest('hex');
            headers['x-workflow-signature'] = signature;
            console.log(`🔐 Added webhook signature header for request ${requestId}`);
        } else {
            console.warn(`⚠️ No webhook secret provided for callback to ${url}`);
        }

        try {
            await axios.post(url, data, {
                headers,
                timeout: 10000
            });

            if (requestId) {
            await db.query(
                `INSERT INTO webhook_deliveries (request_id, url, payload, success)
                 VALUES (?, ?, ?, TRUE)`,
                    [requestId, url, JSON.stringify(data)]
            );
            }
        } catch (error) {
            console.error('Webhook delivery failed:', error.message);
            if (requestId) {
            await db.query(
                `INSERT INTO webhook_deliveries (request_id, url, payload, response_status, success)
                 VALUES (?, ?, ?, ?, FALSE)`,
                    [requestId, url, JSON.stringify(data), error.response?.status || 0]
                );
            }
        }
    }

    fetchRequestRecord = async (requestId) => {
        const [requests] = await db.query(
            'SELECT * FROM approval_requests WHERE id = ?',
            [requestId]
        );
        return requests.length ? requests[0] : null;
    }

    notifyOperationCallback = async (request, decision) => {
        const invocation = await operationModel.getInvocationByRequestId(request.id);
        if (!invocation) {
            return;
        }

        await operationModel.updateInvocationStatus(
            invocation.id,
            decision === 'approved' ? 'approved' : 'rejected'
        );

        const callbackConfig = await operationModel.getCallbackByOperationId(invocation.operation_id);
        if (!callbackConfig) {
            return;
        }

        const targetUrl = decision === 'approved'
            ? callbackConfig.success_url
            : (callbackConfig.failure_url || callbackConfig.success_url);

        if (!targetUrl) {
            return;
        }

        const payload = {
            requestId: request.id,
            invocationId: invocation.id,
            decision,
            status: request.status,
            module: request.module,
            operation: request.operation,
            workflowId: request.workflow_id,
            resourceRef: request.resource_ref,
            correlationId: request.correlation_id,
            payload: safeParseJson(request.payload, request.payload)
        };

        await this.executeCallback({
            url: targetUrl,
            data: payload,
            requestId: request.id,
            secret: callbackConfig.webhook_secret
        });

        await auditLog('operation_decision', 'operation_invocation', invocation.id, null, null, {
            decision,
            requestId: request.id,
            callbackUrl: targetUrl
        });
    }

    evaluateCondition = (conditionExpr, data) => {
        // Simple condition evaluator - in production, use a proper expression engine
        try {
            console.log('🔍 Evaluating condition:', conditionExpr, 'with data:', data);
            // Example: "total_amount > 5000"
            const expr = conditionExpr.replace(/(\w+)/g, 'data.$1');
            console.log('🔍 Transformed expression:', expr);
            const result = eval(expr);
            console.log('🔍 Evaluation result:', result);
            return result;
        } catch (error) {
            console.error('❌ Condition evaluation error:', error);
            return true; // Default to requiring approval
        }
    }

}

module.exports = new WorkflowController();
