const mysql = require('mysql2/promise');
require('dotenv').config();

const createAuditLogsTable = async() => {
    const connection = await mysql.createConnection({
        host: 'localhost',
        user: 'root',
        password: 'root',
        database: 'hybrid_users'
    });

    try {
        console.log('🔧 Creating audit_logs table...');

        // Drop table if it exists to recreate with new structure
        await connection.execute('DROP TABLE IF EXISTS audit_logs');
        console.log('✅ Dropped existing audit_logs table');

        // Create new table with updated structure
        const createTableQuery = `
            CREATE TABLE audit_logs (
                log_id INT PRIMARY KEY AUTO_INCREMENT,
                event_type VARCHAR(100) NOT NULL COMMENT 'Type of event (CREATE, UPDATE, DELETE, READ, LOGIN, etc.)',
                module_name VARCHAR(100) NOT NULL COMMENT 'Module where the event occurred',
                entity_name VARCHAR(100) NOT NULL COMMENT 'Entity type (user, role, permission, etc.)',
                entity_id VARCHAR(50) NULL COMMENT 'ID of the affected entity',
                performed_by VARCHAR(100) NOT NULL COMMENT 'User who performed the action',
                user_role VARCHAR(100) NOT NULL COMMENT 'Role of the user who performed the action',
                action_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'When the action occurred',
                location VARCHAR(255) NULL COMMENT 'Location information',
                ip_address VARCHAR(45) NULL COMMENT 'IP address of the user',
                device_info TEXT NULL COMMENT 'Device and browser information',
                old_data JSON NULL COMMENT 'Previous data (for updates)',
                new_data JSON NULL COMMENT 'New data (for creates/updates)',
                additional_info JSON NULL COMMENT 'Additional context information',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                
                INDEX idx_event_type (event_type),
                INDEX idx_module_name (module_name),
                INDEX idx_entity_name (entity_name),
                INDEX idx_performed_by (performed_by),
                INDEX idx_user_role (user_role),
                INDEX idx_action_time (action_time),
                INDEX idx_entity (entity_name, entity_id),
                INDEX idx_module_event (module_name, event_type)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Audit trail for all system activities'
        `;

        await connection.execute(createTableQuery);
        console.log('✅ audit_logs table created successfully!');

        // Also create a logs table for backward compatibility
        console.log('🔧 Creating logs table for backward compatibility...');
        const createLogsTableQuery = `
            CREATE TABLE IF NOT EXISTS logs (
                id INT PRIMARY KEY AUTO_INCREMENT,
                event_type VARCHAR(100) NOT NULL COMMENT 'Type of event (CREATE, UPDATE, DELETE, READ, LOGIN, etc.)',
                module_name VARCHAR(100) NOT NULL COMMENT 'Module where the event occurred',
                entity_name VARCHAR(100) NOT NULL COMMENT 'Entity type (user, role, permission, etc.)',
                entity_id VARCHAR(50) NULL COMMENT 'ID of the affected entity',
                performed_by VARCHAR(100) NOT NULL COMMENT 'User who performed the action',
                user_role VARCHAR(100) NOT NULL COMMENT 'Role of the user who performed the action',
                action_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'When the action occurred',
                location VARCHAR(255) NULL COMMENT 'Location information',
                ip_address VARCHAR(45) NULL COMMENT 'IP address of the user',
                device_info TEXT NULL COMMENT 'Device and browser information',
                old_data JSON NULL COMMENT 'Previous data (for updates)',
                new_data JSON NULL COMMENT 'New data (for creates/updates)',
                additional_info JSON NULL COMMENT 'Additional context information',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                
                INDEX idx_event_type (event_type),
                INDEX idx_module_name (module_name),
                INDEX idx_entity_name (entity_name),
                INDEX idx_performed_by (performed_by),
                INDEX idx_user_role (user_role),
                INDEX idx_action_time (action_time),
                INDEX idx_entity (entity_name, entity_id),
                INDEX idx_module_event (module_name, event_type)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Logs table for backward compatibility'
        `;

        await connection.execute(createLogsTableQuery);
        console.log('✅ logs table created successfully!');

        console.log('🎉 Audit logs tables creation completed successfully!');

    } catch (error) {
        console.error('❌ Error creating audit logs tables:', error);
    } finally {
        await connection.end();
    }
};

// Run the creation
createAuditLogsTable();