const mysql = require('mysql2/promise');

const createChatTables = async() => {
    const connection = await mysql.createConnection({
        host: 'localhost',
        user: 'root',
        password: 'root',
        database: 'hybrid_users'
    });

    try {
        console.log('🔧 Creating chat tables...');

        // Create conversations table
        const createConversationsTable = `
            CREATE TABLE IF NOT EXISTS conversations (
                id INT PRIMARY KEY AUTO_INCREMENT,
                title VARCHAR(255) NOT NULL,
                type ENUM('direct', 'group') DEFAULT 'direct',
                created_by INT NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                
                INDEX idx_created_by (created_by),
                INDEX idx_type (type),
                INDEX idx_created_at (created_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='Chat conversations'
        `;

        await connection.execute(createConversationsTable);
        console.log('✅ Created conversations table');

        // Create conversation_participants table
        const createParticipantsTable = `
            CREATE TABLE IF NOT EXISTS conversation_participants (
                id INT PRIMARY KEY AUTO_INCREMENT,
                conversation_id INT NOT NULL,
                user_id INT NOT NULL,
                role ENUM('admin', 'member') DEFAULT 'member',
                joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                last_read_at TIMESTAMP NULL,
                
                UNIQUE KEY unique_conversation_user (conversation_id, user_id),
                INDEX idx_conversation_id (conversation_id),
                INDEX idx_user_id (user_id),
                INDEX idx_last_read (last_read_at),
                
                FOREIGN KEY (conversation_id) REFERENCES conversations(id) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='Conversation participants'
        `;

        await connection.execute(createParticipantsTable);
        console.log('✅ Created conversation_participants table');

        // Create messages table
        const createMessagesTable = `
            CREATE TABLE IF NOT EXISTS messages (
                id INT PRIMARY KEY AUTO_INCREMENT,
                conversation_id INT NOT NULL,
                sender_id INT NOT NULL,
                content TEXT NOT NULL,
                message_type ENUM('text', 'image', 'file', 'system') DEFAULT 'text',
                file_url VARCHAR(500) NULL,
                file_name VARCHAR(255) NULL,
                file_size INT NULL,
                is_edited BOOLEAN DEFAULT FALSE,
                edited_at TIMESTAMP NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                
                INDEX idx_conversation_id (conversation_id),
                INDEX idx_sender_id (sender_id),
                INDEX idx_created_at (created_at),
                INDEX idx_message_type (message_type),
                
                FOREIGN KEY (conversation_id) REFERENCES conversations(id) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='Chat messages'
        `;

        await connection.execute(createMessagesTable);
        console.log('✅ Created messages table');

        console.log('🎉 Chat tables creation completed successfully!');

    } catch (error) {
        console.error('❌ Error creating chat tables:', error);
    } finally {
        await connection.end();
    }
};

// Run the creation
createChatTables();