// =============================================
// COMPLETE FRONTEND ROUTING LOGIC UPDATE
// =============================================

// 1. UPDATE authStore.js (Pinia/Vuex Store)
export const useAuthStore = defineStore('auth', {
  state: () => ({
    user: null,
    token: null,
    isAuthenticated: false,
    moduleName: null
  }),
  
  actions: {
    // ✅ FIXED: Use module from API response
    async login(credentials) {
      try {
        const response = await authApi.login(credentials);
        const user = response.data.user;
        
        // Store user data including module
        this.user = user;
        this.token = response.data.token;
        this.isAuthenticated = true;
        this.moduleName = user.module_name; // Store module name
        
        // Get correct route based on module
        const moduleRoute = this.getModuleRoute(user.module_name);
        
        // Navigate to correct module
        navigate(moduleRoute);
        
        return response;
      } catch (error) {
        throw error;
      }
    },

    // ✅ ADD THIS: Module route mapping
    getModuleRoute(moduleName) {
      const moduleRoutes = {
        'supply_chain': '/dashboard/supplyChain',
        'supply_chain_operations': '/dashboard/supplyChainOps',
        'accounts': '/dashboard/accounts',
        'dms': '/dashboard/dms',
        'audit': '/dashboard/audit',
        'hr': '/dashboard/hr',
        'cr': '/dashboard/cr',
        'warehouse': '/dashboard/warehouse'
      };
      
      return moduleRoutes[moduleName] || '/dashboard/supplyChain';
    },

    logout() {
      this.user = null;
      this.token = null;
      this.isAuthenticated = false;
      this.moduleName = null;
      navigate('/login');
    }
  }
});

// 2. UPDATE ROUTER GUARDS (router/index.js or similar)
router.beforeEach((to, from, next) => {
  const authStore = useAuthStore();
  const isAuthenticated = authStore.isAuthenticated;
  const user = authStore.user;
  
  // If authenticated and going to home page, redirect to user's module
  if (isAuthenticated && to.path === '/') {
    const moduleRoute = authStore.getModuleRoute(user?.module_name);
    next(moduleRoute);
  } 
  // If trying to access protected route without auth, redirect to login
  else if (!isAuthenticated && to.meta.requiresAuth) {
    next('/login');
  }
  // Otherwise, allow navigation
  else {
    next();
  }
});

// 3. UPDATE LoginPage.vue (or wherever login happens)
const handleLogin = async () => {
  try {
    setLoading(true);
    
    const response = await authStore.login({
      email: form.email,
      password: form.password
    });
    
    // ✅ FIXED: Store is automatically handled in authStore.login()
    // No need to manually navigate - authStore handles it
    
    notifySuccess('Login successful!');
    
  } catch (error) {
    notifyError('Login failed: ' + error.message);
  } finally {
    setLoading(false);
  }
};

// 4. UPDATE NAVIGATION COMPONENTS
// In your navigation/sidebar component:
const authStore = useAuthStore();
const currentModule = computed(() => authStore.moduleName);

// Show different menu items based on module
const menuItems = computed(() => {
  switch (currentModule.value) {
    case 'accounts':
      return ['Dashboard', 'Invoices', 'Payments', 'Reports'];
    case 'dms':
      return ['Documents', 'Upload', 'Search', 'Archive'];
    case 'supply_chain':
      return ['Inventory', 'Orders', 'Suppliers', 'Reports'];
    case 'cr':
      return ['Customers', 'Tickets', 'Reports', 'Settings'];
    default:
      return ['Dashboard'];
  }
});
