const jwt = require('jsonwebtoken');
const userModel = require('../models/userModel');

module.exports = function(req, res, next) {
    // Get token from header
    const token = req.header('Authorization');

    // Check if not token
    if (!token) {
        return res.status(401).json({ msg: 'No token, authorization denied' });
    }

    // Verify token
    try {
        const decoded = jwt.verify(token.split(' ')[1], process.env.AUTH_JWT_SECRET);

        // Handle two different JWT structures:
        // 1. Super admin: { user: { id, username, isSuperAdmin } }
        // 2. Regular user: { id, role_id, module_id, store_id }
        if (decoded.user) {
            // Super admin format
            req.user = decoded.user;
            req.user.isSuperAdmin = true; // Ensure flag is set
        } else if (decoded.id) {
            // Regular user format - get additional info from database
            req.user = {
                id: decoded.id,
                user_id: decoded.id,
                userId: decoded.id,
                role_id: decoded.role_id,
                module_id: decoded.module_id,
                store_id: decoded.store_id,
                isSuperAdmin: false // Regular users are not super admins
            };

            // Add additional context for multi-store functionality
            req.user.canManagePrices = false; // Will be checked per module
            req.user.canViewAllPrices = (decoded.store_id === 1); // HQTRS can view all prices
        } else {
            // Unknown format
            console.error('Unknown JWT token structure:', decoded);
            return res.status(401).json({ msg: 'Invalid token structure' });
        }

        next();
    } catch (err) {
        console.error('JWT verification error:', err.message);
        res.status(401).json({ msg: 'Token is not valid' });
    }
};