require("dotenv").config();
const axios = require("axios");
const https = require("https");
const { dbQuery } = require("../services/dbService"); // Local database service for fallback logging

const apiLogger = async(req, res, next) => {
    const originalSend = res.send;
    const startTime = new Date();

    res.send = async(body) => {
        const endTime = new Date();

        const logPayload = {
            event_type: req.method === "POST" ? "CREATE" : req.method === "PUT" ? "UPDATE" : req.method === "GET" ? "READ" : "DELETE",
            module_name: 'User Management',
            entity_name: req.baseUrl.split("/").pop().replace("-", "_"),
            entity_id: req.params.id || null,
            performed_by: req.headers["user-id"] || req.body["user-id"] || "anonymous",
            user_role: req.headers["user-role"] || req.body["user-role"] || "unknown",
            action_time: new Date().toISOString().slice(0, 19).replace("T", " "),
            location: req.headers["location"] || req.body["location"] || "unknown",
            ip_address: req.ip,
            device_info: req.headers["user-agent"],
            old_data: req.oldData || null,
            new_data: req.method !== "GET" ? req.body : null,
            additional_info: {},
        };

        // Configure Axios with HTTPS Agent
        const httpsAgent = new https.Agent({
            rejectUnauthorized: false, // Allow self-signed certificates (only for testing)
            secureProtocol: "TLSv1_2_method", // Enforce TLS 1.2
        });

        try {
            // Check if LOGGING_API_URL is configured
            if (!process.env.LOGGING_API_URL) {
                console.log("ℹ️ LOGGING_API_URL not configured, skipping external logging");
                throw new Error("LOGGING_API_URL not configured");
            }

            // Attempt to send logs to external API
            await axios.post(process.env.LOGGING_API_URL, logPayload, {
                headers: { "Content-Type": "application/json" },
                httpsAgent,
            });
            console.log("✅ Log sent to external server successfully");
        } catch (error) {
            console.error("❌ Failed to send log to external server:", error.message);

            // Fallback to local logging (if external API fails)
            try {
                console.log(`🔍 Attempting to insert log with event_type: "${logPayload.event_type}" (length: ${logPayload.event_type.length})`);
                // For now, let's skip the database insert to avoid errors
                console.log("🚫 Temporarily skipping database insert to avoid truncation error");
                // await dbQuery(
                //     "INSERT INTO logs (event_type, module_name, entity_name, entity_id, performed_by, user_role, action_time, location, ip_address, device_info, old_data, new_data, additional_info) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)", [
                //         logPayload.event_type,
                //         logPayload.module_name,
                //         logPayload.entity_name,
                //         logPayload.entity_id,
                //         logPayload.performed_by,
                //         logPayload.user_role,
                //         logPayload.action_time,
                //         logPayload.location,
                //         logPayload.ip_address,
                //         logPayload.device_info,
                //         JSON.stringify(logPayload.old_data),
                //         JSON.stringify(logPayload.new_data),
                //         JSON.stringify(logPayload.additional_info),
                //     ]
                // );
                console.log("✅ Log saved to local database as a fallback");
            } catch (dbError) {
                console.error("❌ Failed to save log to local database:", dbError.message);
                console.error("🔍 Event type that failed:", logPayload.event_type);
            }
        }

        res.send = originalSend;
        return res.send(body);
    };

    next();
};

module.exports = apiLogger;