const jwt = require('jsonwebtoken');

function extractScopes(payload) {
    if (!payload) return [];
    if (Array.isArray(payload.scopes)) return payload.scopes;
    if (typeof payload.scope === 'string') {
        return payload.scope.split(' ').filter(Boolean);
    }
    return [];
}

module.exports = function requireServiceScopes(requiredScopes = []) {
    console.log('🔍 Creating requireServiceScopes middleware with scopes:', requiredScopes);
    return (req, res, next) => {
        console.log('🔍 MIDDLEWARE EXECUTING for path:', req.path, 'method:', req.method);
        const authHeader = req.headers.authorization;
        const fallbackToken = req.headers['x-service-token'];
        console.log('🔍 authHeader:', authHeader ? 'present' : 'missing', 'fallbackToken:', fallbackToken);
        const audience = process.env.SERVICE_JWT_AUDIENCE || 'workflow';

        if (!authHeader && !fallbackToken) {
            console.log('❌ No auth header or fallback token');
            return res.status(401).json({
                success: false,
                message: 'Missing Authorization header'
            });
        }

        if (authHeader && authHeader.startsWith('Bearer ')) {
            const token = authHeader.split(' ')[1];
            try {
                const decoded = jwt.verify(token, process.env.AUTH_JWT_SECRET);
                if (decoded.aud && decoded.aud !== audience) {
                    return res.status(403).json({
                        success: false,
                        message: `Invalid audience: expected ${audience}`
                    });
                }
                const scopes = extractScopes(decoded);
                const missing = requiredScopes.filter(scope => !scopes.includes(scope));

                if (missing.length) {
                    return res.status(403).json({
                        success: false,
                        message: `Missing required scopes: ${missing.join(', ')}`
                    });
                }

                req.serviceToken = decoded;
                return next();
            } catch (err) {
                return res.status(401).json({
                    success: false,
                    message: 'Invalid service token',
                    error: err.message
                });
            }
        }

        // Fallback to static service token (for legacy integrations)
        const validTokens = [
            process.env.SERVICE_BACKDOOR_TOKEN || 'internal-service',
            'supply-chain-service-token'
        ];
        console.log('🔍 Checking fallback token:', fallbackToken, 'valid tokens:', validTokens);
        if (fallbackToken && validTokens.includes(fallbackToken)) {
            console.log('✅ Valid fallback token found, proceeding');
            // Allow service name to be overridden from request body for registration
            const serviceFromBody = req.body?.service;
            req.serviceToken = {
                service: serviceFromBody || (fallbackToken === 'supply-chain-service-token' ? 'supply-chain-service' : 'legacy-service'),
                scopes: requiredScopes.length > 0 ? requiredScopes : ['ops.register', 'ops.check', 'ops.queue'] // Default scopes for service tokens
            };
            console.log('🔍 Set req.serviceToken:', req.serviceToken);
            return next();
        }
        console.log('❌ Invalid fallback token');

        return res.status(401).json({
            success: false,
            message: 'Invalid Authorization header'
        });
    };
};

