const db = require('../config/db');

class ChatModel {
    // Create a new conversation
    static async createConversation(participants, title = null, type = 'group') {
        const query = `
            INSERT INTO conversations (title, type, created_at, updated_at)
            VALUES (?, ?, NOW(), NOW())
        `;

        try {
            const [result] = await db.execute(query, [title, type]);
            const conversationId = result.insertId;

            // Add participants to conversation
            for (const participantId of participants) {
                await this.addParticipantToConversation(conversationId, participantId);
            }

            return conversationId;
        } catch (error) {
            throw error;
        }
    }

    // Add participant to conversation
    static async addParticipantToConversation(conversationId, userId) {
        const query = `
            INSERT INTO conversation_participants (conversation_id, user_id, joined_at)
            VALUES (?, ?, NOW())
        `;

        try {
            await db.execute(query, [conversationId, userId]);
        } catch (error) {
            throw error;
        }
    }

    // Get conversations for a user
    static async getUserConversations(userId) {
        const query = `
            SELECT 
                c.id,
                c.title,
                c.type,
                c.created_at,
                c.updated_at,
                COUNT(cp.user_id) as participant_count,
                (SELECT message_content FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message,
                (SELECT created_at FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message_time,
                (SELECT u2.name FROM messages m2 INNER JOIN users u2 ON m2.sender_id = u2.id WHERE m2.conversation_id = c.id ORDER BY m2.created_at DESC LIMIT 1) as last_message_sender_name,
                (SELECT m2.sender_id FROM messages m2 WHERE m2.conversation_id = c.id ORDER BY m2.created_at DESC LIMIT 1) as last_message_sender_id,
                COALESCE((
                    SELECT COUNT(*)
                    FROM messages m
                    WHERE m.conversation_id = c.id
                    AND m.created_at > COALESCE(cp.last_read_at, '1970-01-01')
                ), 0) as unread_count
            FROM conversations c
            INNER JOIN conversation_participants cp ON c.id = cp.conversation_id
            WHERE cp.user_id = ?
            GROUP BY c.id, cp.last_read_at
            ORDER BY c.updated_at DESC
        `;

        try {
            const [rows] = await db.execute(query, [userId]);
            return rows;
        } catch (error) {
            throw error;
        }
    }

    // Get conversation participants
    static async getConversationParticipants(conversationId) {
        const query = `
            SELECT 
                u.id,
                u.name,
                u.username,
                u.email,
                cp.joined_at
            FROM conversation_participants cp
            INNER JOIN users u ON cp.user_id = u.id
            WHERE cp.conversation_id = ?
            ORDER BY cp.joined_at ASC
        `;

        try {
            const [rows] = await db.execute(query, [conversationId]);
            return rows;
        } catch (error) {
            throw error;
        }
    }

    // Save a message
    static async saveMessage(conversationId, senderId, message, messageType = 'text') {
        const query = `
            INSERT INTO messages (conversation_id, sender_id, message_content, message_type, created_at)
            VALUES (?, ?, ?, ?, NOW())
        `;

        try {
            const [result] = await db.execute(query, [conversationId, senderId, message, messageType]);

            // Update conversation last activity
            await this.updateConversationActivity(conversationId);

            return result.insertId;
        } catch (error) {
            throw error;
        }
    }

    // Get messages for a conversation with read status
    static async getConversationMessages(conversationId, userId, limit = 50, offset = 0) {
        const query = `
            SELECT 
                m.id,
                m.message_content as message,
                m.message_type,
                m.created_at,
                u.id as sender_id,
                u.name as sender_name,
                u.username as sender_username,
                m.file_name,
                m.file_size,
                m.file_url,
                m.file_mime_type,
                m.file_thumbnail_path,
                CASE
                    WHEN m.file_thumbnail_path IS NOT NULL AND m.file_thumbnail_path != '' THEN 
                        CONCAT('/files/thumbnails/', SUBSTRING_INDEX(m.file_thumbnail_path, '/', -1))
                    ELSE NULL
                END AS file_thumbnail_url,
                CASE
                    WHEN m.sender_id = ? THEN
                        -- For messages sent by current user, check if any other participant has read it
                        CASE WHEN EXISTS(
                            SELECT 1
                            FROM conversation_participants cp2
                            WHERE cp2.conversation_id = m.conversation_id
                            AND cp2.user_id != ?
                            AND cp2.last_read_at >= m.created_at
                        ) THEN 'read'
                        WHEN EXISTS(
                            SELECT 1
                            FROM conversation_participants cp3
                            WHERE cp3.conversation_id = m.conversation_id
                            AND cp3.user_id != ?
                        ) THEN 'delivered'
                        ELSE 'sent' END
                    WHEN m.created_at <= COALESCE(cp.last_read_at, '1970-01-01') THEN 'read'
                    ELSE 'delivered'
                END as status
            FROM messages m
            INNER JOIN users u ON m.sender_id = u.id
            LEFT JOIN conversation_participants cp ON cp.conversation_id = m.conversation_id AND cp.user_id = ?
            WHERE m.conversation_id = ?
            ORDER BY m.created_at DESC
            LIMIT ? OFFSET ?
        `;

        try {
            const [rows] = await db.execute(query, [userId, userId, userId, userId, conversationId, limit, offset]);
            return rows.reverse(); // Return in chronological order
        } catch (error) {
            throw error;
        }
    }

    // Get user by ID
    static async getUserById(userId) {
        const query = `
            SELECT id, name, username, email
            FROM users
            WHERE id = ?
        `;

        try {
            const [rows] = await db.execute(query, [userId]);
            return rows[0];
        } catch (error) {
            throw error;
        }
    }

    // Update conversation activity
    static async updateConversationActivity(conversationId) {
        const query = `
            UPDATE conversations 
            SET updated_at = NOW() 
            WHERE id = ?
        `;

        try {
            await db.execute(query, [conversationId]);
        } catch (error) {
            throw error;
        }
    }

    // Check if user is participant in conversation
    static async isUserInConversation(userId, conversationId) {
        const query = `
            SELECT COUNT(*) as count
            FROM conversation_participants
            WHERE user_id = ? AND conversation_id = ?
        `;

        try {
            const [rows] = await db.execute(query, [userId, conversationId]);
            return rows[0].count > 0;
        } catch (error) {
            throw error;
        }
    }

    // Get unread message count for user in conversation
    static async getUnreadCount(userId, conversationId) {
        const query = `
            SELECT COUNT(*) as count
            FROM messages m
            WHERE m.conversation_id = ? 
            AND m.created_at > (
                SELECT COALESCE(MAX(last_read_at), '1970-01-01')
                FROM conversation_participants
                WHERE user_id = ? AND conversation_id = ?
            )
        `;

        try {
            const [rows] = await db.execute(query, [conversationId, userId, conversationId]);
            return rows[0].count;
        } catch (error) {
            throw error;
        }
    }

    // Mark messages as read
    static async markMessagesAsRead(userId, conversationId) {
        const query = `
            UPDATE conversation_participants
            SET last_read_at = NOW()
            WHERE user_id = ? AND conversation_id = ?
        `;

        try {
            await db.execute(query, [userId, conversationId]);
        } catch (error) {
            throw error;
        }
    }

    static async saveFileMessage({
        conversationId,
        senderId,
        message,
        messageType,
        fileName,
        fileSize,
        fileUrl,
        filePath,
        fileMimeType,
        fileThumbnailPath
    }) {
        const query = `
            INSERT INTO messages (
                conversation_id,
                sender_id,
                message_content,
                message_type,
                file_name,
                file_size,
                file_url,
                file_path,
                file_mime_type,
                file_thumbnail_path
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `;

        try {
            const [result] = await db.execute(query, [
                conversationId,
                senderId,
                message,
                messageType,
                fileName,
                fileSize,
                fileUrl,
                filePath,
                fileMimeType,
                fileThumbnailPath
            ]);

            return result.insertId;
        } catch (error) {
            throw error;
        }
    }

    static async getMessageById(messageId) {
        const query = `
            SELECT
                m.id,
                m.conversation_id as conversationId,
                m.sender_id as sender_id,
                u.name as sender_name,
                u.username as sender_username,
                m.message_content as message,
                m.message_type,
                m.status,
                m.file_name,
                m.file_size,
                m.file_url,
                m.file_path,
                m.file_mime_type,
                m.file_thumbnail_path,
                m.created_at
            FROM messages m
            INNER JOIN users u ON m.sender_id = u.id
            WHERE m.id = ?
        `;

        try {
            const [rows] = await db.execute(query, [messageId]);
            return rows[0];
        } catch (error) {
            throw error;
        }
    }

    static async getUserById(userId) {
        const query = `
            SELECT id, name, username, email
            FROM users
            WHERE id = ?
        `;
        try {
            const [rows] = await db.execute(query, [userId]);
            return rows[0];
        } catch (error) {
            throw error;
        }
    }
}

module.exports = ChatModel;