const db = require('../config/db');

function parseJson(value) {
    if (!value) return null;
    if (typeof value === 'object') return value;
    try {
        return JSON.parse(value);
    } catch (err) {
        return null;
    }
}

async function getOperationById(id) {
    const [rows] = await db.query('SELECT * FROM operations WHERE id = ?', [id]);
    if (!rows.length) return null;
    const operation = rows[0];
    operation.payload_schema = parseJson(operation.payload_schema);
    return operation;
}

async function getOperationBySignature({ service, module, operation, method, path }) {
    const [rows] = await db.query(
        `SELECT * FROM operations
         WHERE service = ? AND module = ? AND operation = ? AND method = ? AND path = ?
         LIMIT 1`,
        [service, module, operation, method, path]
    );
    if (!rows.length) return null;
    const operationRecord = rows[0];
    operationRecord.payload_schema = parseJson(operationRecord.payload_schema);
    return operationRecord;
}

async function upsertOperation(operation) {
    const {
        service,
        module,
        operation: opName,
        method,
        path,
        version = 'v1',
        payloadSchema = null,
        isActive = true,
        createdBy = 'system'
    } = operation;

    await db.query(
        `INSERT INTO operations
            (service, module, operation, method, path, version, payload_schema, is_active, created_by)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
         ON DUPLICATE KEY UPDATE
            version = VALUES(version),
            payload_schema = VALUES(payload_schema),
            is_active = VALUES(is_active),
            updated_at = CURRENT_TIMESTAMP`,
        [
            service,
            module,
            opName,
            method,
            path,
            version,
            payloadSchema ? JSON.stringify(payloadSchema) : null,
            isActive,
            createdBy
        ]
    );

    return getOperationBySignature({
        service,
        module,
        operation: opName,
        method,
        path
    });
}

async function listOperations(filters = {}) {
    const conditions = [];
    const params = [];

    if (filters.service) {
        conditions.push('o.service = ?');
        params.push(filters.service);
    }
    if (filters.module) {
        conditions.push('o.module = ?');
        params.push(filters.module);
    }
    if (filters.operation) {
        conditions.push('o.operation = ?');
        params.push(filters.operation);
    }
    if (filters.isActive !== undefined) {
        conditions.push('o.is_active = ?');
        params.push(!!filters.isActive);
    }

    let query = `
        SELECT o.*, oc.success_url, oc.failure_url, oc.webhook_secret, oc.retry_policy
        FROM operations o
        LEFT JOIN operation_callbacks oc ON o.id = oc.operation_id
    `;

    if (conditions.length) {
        query += ' WHERE ' + conditions.join(' AND ');
    }

    query += ' ORDER BY o.created_at DESC';

    const [rows] = await db.query(query, params);
    return rows.map(row => {
        const safeRow = {
            ...row,
            payload_schema: parseJson(row.payload_schema),
            retry_policy: parseJson(row.retry_policy)
        };
        delete safeRow.webhook_secret;
        return safeRow;
    });
}

async function upsertCallback(operationId, callback) {
    if (!callback) return null;

    await db.query(
        `INSERT INTO operation_callbacks (operation_id, success_url, failure_url, webhook_secret, retry_policy)
         VALUES (?, ?, ?, ?, ?)
         ON DUPLICATE KEY UPDATE
            success_url = VALUES(success_url),
            failure_url = VALUES(failure_url),
            webhook_secret = VALUES(webhook_secret),
            retry_policy = VALUES(retry_policy),
            updated_at = CURRENT_TIMESTAMP`,
        [
            operationId,
            callback.success_url || null,
            callback.failure_url || null,
            callback.webhook_secret,
            callback.retry_policy ? JSON.stringify(callback.retry_policy) : null
        ]
    );

    return getCallbackByOperationId(operationId);
}

async function getCallbackByOperationId(operationId) {
    const [rows] = await db.query(
        'SELECT * FROM operation_callbacks WHERE operation_id = ? LIMIT 1',
        [operationId]
    );
    if (!rows.length) return null;
    const callback = rows[0];
    callback.retry_policy = parseJson(callback.retry_policy);
    return callback;
}

async function recordInvocation(invocation) {
    const payloadJson = typeof invocation.payload === 'string'
        ? invocation.payload
        : JSON.stringify(invocation.payload || {});

    const [result] = await db.query(
        `INSERT INTO operation_invocations
            (operation_id, approval_request_id, correlation_id, resource_ref, payload, created_by, status, idempotency_key)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
        [
            invocation.operationId,
            invocation.approvalRequestId || null,
            invocation.correlationId || null,
            invocation.resourceRef || null,
            payloadJson,
            invocation.createdBy || null,
            invocation.status || 'pending',
            invocation.idempotencyKey || null
        ]
    );

    return getInvocationById(result.insertId);
}

async function getInvocationById(id) {
    const [rows] = await db.query('SELECT * FROM operation_invocations WHERE id = ?', [id]);
    if (!rows.length) return null;
    const invocation = rows[0];
    invocation.payload = parseJson(invocation.payload);
    return invocation;
}

async function findInvocationByIdempotency(operationId, idempotencyKey) {
    if (!idempotencyKey) return null;
    const [rows] = await db.query(
        `SELECT * FROM operation_invocations
         WHERE operation_id = ? AND idempotency_key = ?
         LIMIT 1`,
        [operationId, idempotencyKey]
    );
    if (!rows.length) return null;
    const invocation = rows[0];
    invocation.payload = parseJson(invocation.payload);
    return invocation;
}

async function getInvocationByRequestId(requestId) {
    const [rows] = await db.query(
        `SELECT * FROM operation_invocations
         WHERE approval_request_id = ?
         LIMIT 1`,
        [requestId]
    );
    if (!rows.length) return null;
    const invocation = rows[0];
    invocation.payload = parseJson(invocation.payload);
    return invocation;
}

async function updateInvocationStatus(id, status) {
    await db.query(
        `UPDATE operation_invocations
         SET status = ?, updated_at = CURRENT_TIMESTAMP
         WHERE id = ?`,
        [status, id]
    );
    return getInvocationById(id);
}

module.exports = {
    upsertOperation,
    listOperations,
    upsertCallback,
    getOperationById,
    getOperationBySignature,
    getCallbackByOperationId,
    recordInvocation,
    findInvocationByIdempotency,
    getInvocationByRequestId,
    updateInvocationStatus,
    getInvocationById
};

