const db = require('../config/db');

// Get all users with role and module info
const getAllUsers = async () => {
    const [users] = await db.execute(`
        SELECT 
            u.id,
            u.name,
            u.email,
            u.username,
            u.role_id,
            r.name AS role_name,
            u.store_id,
            u.module_id,
            m.name AS module_name,
            u.is_blocked
        FROM users u
        LEFT JOIN roles r ON u.role_id = r.id
        LEFT JOIN modules m ON u.module_id = m.id
        ORDER BY u.name ASC
    `);
    return users;
};

// Get a user by ID with more details
const getUserById = async(id) => {
    const [users] = await db.execute(`
        SELECT 
            u.id, 
            u.name, 
            u.email, 
            u.username, 
            u.role_id, 
            u.module_id, 
            u.store_id, 
            u.phone_number, 
            u.is_blocked,
            u.password,
            r.name as role_name,
            m.name as module_name,
            s.name as store_name
        FROM users u
        LEFT JOIN roles r ON u.role_id = r.id
        LEFT JOIN modules m ON u.module_id = m.id
        LEFT JOIN stores s ON u.store_id = s.id
        WHERE u.id = ?
    `, [id]);
    return users[0];
};

// Create a new user
const createUser = async (name, email, password, username, role, module, phone_number, store_id) => {
    try {
        const [result] = await db.execute(
            "INSERT INTO users (name, email, password, username, role_id, module_id, phone_number, store_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
            [name, email, password, username, role, module, phone_number, store_id]
        );
        return result.insertId;
    } catch (error) {
        console.error("Error creating user:", error);
        throw error;
    }
};

// Update user details
const updateUser = async (id, name, email, role, module, phone_number, store_id) => {
    await db.execute(
        "UPDATE users SET name = ?, email = ?, role_id = ?, module_id = ?, phone_number = ?, store_id = ? WHERE id = ?",
        [name, email, role, module, phone_number, store_id, id]
    );
};

// Delete a user
const deleteUser = async(id) => {
    await db.execute("DELETE FROM users WHERE id = ?", [id]);
};

// Get a user by email
const findByEmail = async(email) => {
    const [rows] = await db.execute(`
        SELECT 
            u.*, 
            m.name as module_name, 
            s.name as store_name
        FROM users u
        LEFT JOIN modules m ON u.module_id = m.id
        LEFT JOIN stores s ON u.store_id = s.id
        WHERE u.email = ?
    `, [email]);
    return rows[0];
};

// Get users by IDs
const getUsersByIds = async(ids) => {
    console.log('userModel.getUsersByIds: Received IDs:', ids); // Log received IDs
    if (!ids || ids.length === 0) {
        console.warn('userModel.getUsersByIds: Empty IDs array.');
        return [];
    }
    const placeholders = ids.map(() => '?').join(',');
    const query = `SELECT id, username FROM users WHERE id IN (${placeholders})`;
    console.log('userModel.getUsersByIds: Executing query:', query, ids); // Log query and params
    const [users] = await db.execute(query, ids);
    console.log('userModel.getUsersByIds: Query result:', users); // Log query result
    return users;
};

module.exports = { getAllUsers, getUserById, createUser, updateUser, deleteUser, findByEmail, getUsersByIds };
