const express = require('express');
const router = express.Router();
const authenticateToken = require('../middleware/authMiddleware');
const serviceAuth = require('../middleware/serviceAuth');
const notificationService = require('../services/notificationService');

// Internal: create and broadcast a notification (for other services)
router.post('/', serviceAuth, async (req, res) => {
    try {
        const { userId, title, message, link, type, priority, icon, sourceModule, payload } = req.body;
        if (!userId || !title || !message) {
            return res.status(400).json({ message: 'userId, title and message are required' });
        }
        const notif = await notificationService.createNotification({ userId, title, message, link, type, priority, icon, sourceModule, payload });
        res.json({ success: true, data: notif });
    } catch (err) {
        console.error('Error creating notification:', err);
        res.status(500).json({ message: 'Failed to create notification' });
    }
});

// List current user notifications
router.get('/', authenticateToken, async (req, res) => {
    try {
        // Extract user ID with fallback
        const userId = req.user?.id || req.user?.user_id || req.user?.userId;
        
        if (!userId) {
            console.error('No user ID found in token:', req.user);
            return res.status(400).json({ message: 'User ID not found in token' });
        }

        const { unreadOnly, limit, offset } = req.query;
        const rows = await notificationService.listNotifications(userId, {
            unreadOnly: unreadOnly === 'true',
            limit: limit ? Number(limit) : 20,
            offset: offset ? Number(offset) : 0,
        });
        res.json({ success: true, data: rows });
    } catch (err) {
        console.error('Error listing notifications:', err);
        res.status(500).json({ message: 'Failed to list notifications' });
    }
});

// Unread count
router.get('/unread-count', authenticateToken, async (req, res) => {
    try {
        // Extract user ID with fallback
        const userId = req.user?.id || req.user?.user_id || req.user?.userId;
        
        if (!userId) {
            console.error('No user ID found in token:', req.user);
            return res.status(400).json({ message: 'User ID not found in token' });
        }

        const count = await notificationService.getUnreadCount(userId);
        res.json({ success: true, data: { count } });
    } catch (err) {
        console.error('Error getting unread count:', err);
        res.status(500).json({ message: 'Failed to get unread count' });
    }
});

// Mark read by id
router.patch('/:id/read', authenticateToken, async (req, res) => {
    try {
        // Extract user ID with fallback
        const userId = req.user?.id || req.user?.user_id || req.user?.userId;
        
        if (!userId) {
            console.error('No user ID found in token:', req.user);
            return res.status(400).json({ message: 'User ID not found in token' });
        }

        await notificationService.markRead(userId, req.params.id);
        res.json({ success: true });
    } catch (err) {
        console.error('Error marking read:', err);
        res.status(500).json({ message: 'Failed to mark notification as read' });
    }
});

// Mark all read
router.post('/mark-all-read', authenticateToken, async (req, res) => {
    try {
        // Extract user ID with fallback
        const userId = req.user?.id || req.user?.user_id || req.user?.userId;
        
        if (!userId) {
            console.error('No user ID found in token:', req.user);
            return res.status(400).json({ message: 'User ID not found in token' });
        }

        await notificationService.markAllRead(userId);
        res.json({ success: true });
    } catch (err) {
        console.error('Error marking all read:', err);
        res.status(500).json({ message: 'Failed to mark all as read' });
    }
});

module.exports = router;












