const express = require('express');
const router = express.Router();
const workflowController = require('../controllers/workflowController');
const apiLogger = require('../middleware/logsMiddleware');
const requireServiceScopes = require('../middleware/serviceScopeAuth');
const authenticateToken = require('../middleware/authMiddleware');

console.log('🔍 Workflow routes loaded');

// ========================================
// HEALTH
// ========================================
router.get('/health', async (req, res) => {
    return workflowController.health(req, res);
});

// ========================================
// WORKFLOW MANAGEMENT (Admin-facing)
// ========================================

// Create workflow
router.post('/workflows', apiLogger, workflowController.createWorkflow);

// Get workflows
router.get('/workflows', apiLogger, (req, res) => {
    console.log('🔍 /workflows route hit');
    return workflowController.getWorkflows(req, res);
});

// Get workflow by ID
router.get('/workflows/:id', apiLogger, workflowController.getWorkflowById);

// Update workflow
router.put('/workflows/:id', apiLogger, workflowController.updateWorkflow);

// Delete workflow
router.delete('/workflows/:id', apiLogger, workflowController.deleteWorkflow);

// Create policy
router.post('/policies', apiLogger, workflowController.createPolicy);

// Get policies
router.get('/policies', apiLogger, workflowController.getPolicies);

// ========================================
// OPERATIONS INTEGRATION (Module-facing)
// ========================================

// Register service operations
router.post(
    '/operations/register',
    apiLogger,
    requireServiceScopes(), // Allow any valid service token
    workflowController.registerOperations
);

// List registered operations
router.get('/operations', apiLogger, workflowController.listOperations);

// Check if approval is required
router.get('/operations/check', apiLogger, workflowController.checkApprovalRequired);

// POST-based policy check (SDK)
router.post(
    '/operations/check',
    apiLogger,
    requireServiceScopes(['ops.check']),
    workflowController.checkOperationPolicy
);

// Queue operation for approval
router.post(
    '/operations/queue',
    apiLogger,
    requireServiceScopes(['ops.queue']),
    workflowController.queueOperation
);

// Admin test runner (uses user token, queues via service context internally)
router.post(
    '/operations/test-run',
    apiLogger,
    authenticateToken,
    workflowController.queueOperationTest
);

// ========================================
// APPROVAL ACTIONS (User-facing)
// ========================================

// Get approval requests
router.get('/requests', apiLogger, workflowController.getRequests);

// Lightweight status endpoint (should be before :id)
router.get('/requests/:id/status', apiLogger, workflowController.getRequestStatus);

// Get request by ID
router.get('/requests/:id', apiLogger, workflowController.getRequestById);

// Approve request
router.post('/requests/:id/approve', apiLogger, workflowController.approveRequest.bind(workflowController));

// Reject request
router.post('/requests/:id/reject', apiLogger, workflowController.rejectRequest.bind(workflowController));

// Get all payloads with logs (for debugging/monitoring)
router.get('/payloads', apiLogger, workflowController.getAllPayloadsWithLogs.bind(workflowController));

// Get detailed logs for a specific payload
router.get('/payloads/:id/logs', apiLogger, workflowController.getWebhookDeliveries.bind(workflowController));

// Get approval actions for a specific payload
router.get('/payloads/:id/actions', apiLogger, workflowController.getApprovalActions.bind(workflowController));

// Callback endpoint for modules (when approval is complete)
router.post('/callbacks/:requestId/complete', apiLogger, (req, res) => {
    // This would be called by the workflow controller after approval
    // For now, just acknowledge
    res.json({ success: true, message: 'Callback received' });
});

// Emit WebSocket events from other services
router.post('/emit-event', requireServiceScopes(), (req, res) => {
    const { event, data, room } = req.body;
    const { emitToRoom } = require('../socket');

    try {
        emitToRoom(room, event, data);
        res.json({ success: true, message: `Event ${event} emitted to room ${room}` });
    } catch (error) {
        console.error('Failed to emit event:', error);
        res.status(500).json({ success: false, message: 'Failed to emit event' });
    }
});

module.exports = router;
