const db = require('../config/db');
const { getAllOperations } = require('../config/crudOperationsConfig');

async function bulkImportCRUDOperations() {
    try {
        console.log('🚀 Starting bulk import of CRUD operations...');

        const operations = getAllOperations();
        console.log(`📋 Found ${operations.length} operations to import`);

        let successCount = 0;
        let errorCount = 0;
        const errors = [];

        for (const operation of operations) {
            try {
                // Check if operation already exists
                const [existing] = await db.query(`
                    SELECT id FROM crud_operations_registry 
                    WHERE module = ? AND operation = ?
                `, [operation.module, operation.operation]);

                if (existing.length > 0) {
                    console.log(`⚠️  Operation ${operation.module}.${operation.operation} already exists, skipping...`);
                    continue;
                }

                // Insert operation
                await db.query(`
                    INSERT INTO crud_operations_registry 
                    (module, operation, entity_type, requires_approval, approval_workflow_id, description, created_by)
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                `, [
                    operation.module,
                    operation.operation,
                    operation.entity_type,
                    operation.requires_approval,
                    operation.approval_workflow_id,
                    operation.description,
                    2 // Default created_by user
                ]);

                console.log(`✅ Imported: ${operation.module}.${operation.operation}`);
                successCount++;

            } catch (error) {
                console.error(`❌ Error importing ${operation.module}.${operation.operation}:`, error.message);
                errors.push({
                    operation: `${operation.module}.${operation.operation}`,
                    error: error.message
                });
                errorCount++;
            }
        }

        console.log('\n📊 Import Summary:');
        console.log(`✅ Successfully imported: ${successCount} operations`);
        console.log(`❌ Failed to import: ${errorCount} operations`);

        if (errors.length > 0) {
            console.log('\n❌ Errors:');
            errors.forEach(error => {
                console.log(`  - ${error.operation}: ${error.error}`);
            });
        }

        console.log('\n🎉 Bulk import completed!');

    } catch (error) {
        console.error('❌ Error during bulk import:', error);
    } finally {
        process.exit(0);
    }
}

// Run the import if this script is executed directly
if (require.main === module) {
    bulkImportCRUDOperations();
}

module.exports = { bulkImportCRUDOperations };