require('dotenv').config({ path: require('path').resolve(__dirname, '../../../.env') });
const fs = require('fs');
const path = require('path');
const db = require('../config/db');

async function createCentralWorkflowTables() {
    try {
        console.log('Creating central workflow tables...');

        const sqlFile = path.join(__dirname, '../create_central_workflow_tables.sql');
        const sql = fs.readFileSync(sqlFile, 'utf8');

        // Parse SQL statements (handle multi-line and comments)
        const statements = [];
        let currentStatement = '';
        let inComment = false;

        const lines = sql.split('\n');

        for (const line of lines) {
            const trimmedLine = line.trim();

            // Skip empty lines and comments
            if (!trimmedLine || trimmedLine.startsWith('--')) {
                continue;
            }

            // Handle multi-line comments
            if (trimmedLine.startsWith('/*')) {
                inComment = true;
                continue;
            }
            if (inComment) {
                if (trimmedLine.endsWith('*/')) {
                    inComment = false;
                }
                continue;
            }

            currentStatement += line + '\n';

            // Check if this line ends a statement
            if (trimmedLine.endsWith(';')) {
                statements.push(currentStatement.trim());
                currentStatement = '';
            }
        }

        console.log(`Found ${statements.length} SQL statements to execute`);

        for (let i = 0; i < statements.length; i++) {
            const statement = statements[i];
            if (statement.trim()) {
                console.log(`Executing statement ${i + 1}:`, statement.trim().substring(0, 100) + '...');
                try {
                    await db.execute(statement);
                    console.log('✅ Statement executed successfully');
                } catch (stmtError) {
                    console.error('❌ Statement failed:', stmtError.message);
                    console.error('Statement was:', statement);
                    throw stmtError;
                }
            }
        }

        console.log('✅ Central workflow tables created successfully');
    } catch (error) {
        console.error('❌ Error creating central workflow tables:', error);
        throw error;
    }
}

// If run directly, execute and exit
if (require.main === module) {
    createCentralWorkflowTables()
        .then(() => process.exit(0))
        .catch(err => {
            console.error(err);
            process.exit(1);
        });
} else {
    module.exports = createCentralWorkflowTables;
}
