require('dotenv').config({ path: require('path').resolve(__dirname, '../../../.env') });
const db = require('../config/db');

const statements = [
    `CREATE TABLE IF NOT EXISTS operations (
        id INT AUTO_INCREMENT PRIMARY KEY,
        service VARCHAR(100) NOT NULL,
        module VARCHAR(100) NOT NULL,
        operation VARCHAR(100) NOT NULL,
        method VARCHAR(10) NOT NULL,
        path VARCHAR(255) NOT NULL,
        version VARCHAR(20) DEFAULT 'v1',
        payload_schema JSON,
        is_active BOOLEAN DEFAULT TRUE,
        created_by VARCHAR(100),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        UNIQUE KEY uniq_service_operation (service, module, operation, method, path),
        INDEX idx_module_operation (module, operation),
        INDEX idx_active (is_active)
    )`,
    `CREATE TABLE IF NOT EXISTS operation_callbacks (
        id INT AUTO_INCREMENT PRIMARY KEY,
        operation_id INT NOT NULL,
        success_url VARCHAR(500),
        failure_url VARCHAR(500),
        webhook_secret VARCHAR(255) NOT NULL,
        retry_policy JSON,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (operation_id) REFERENCES operations(id) ON DELETE CASCADE,
        UNIQUE KEY uniq_operation (operation_id)
    )`,
    `CREATE TABLE IF NOT EXISTS operation_invocations (
        id INT AUTO_INCREMENT PRIMARY KEY,
        operation_id INT NOT NULL,
        approval_request_id INT,
        correlation_id VARCHAR(255),
        resource_ref VARCHAR(255),
        payload JSON,
        created_by VARCHAR(100),
        status ENUM('pending', 'queued', 'approved', 'rejected', 'cancelled') DEFAULT 'pending',
        idempotency_key VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (operation_id) REFERENCES operations(id) ON DELETE CASCADE,
        FOREIGN KEY (approval_request_id) REFERENCES approval_requests(id) ON DELETE SET NULL,
        UNIQUE KEY uniq_idempotency (operation_id, idempotency_key),
        INDEX idx_operation_status (operation_id, status),
        INDEX idx_correlation (correlation_id),
        INDEX idx_resource_ref (resource_ref)
    )`
];

async function createOperationsRegistryTables() {
    console.log('Creating operations registry tables...');
    for (const statement of statements) {
        try {
            await db.execute(statement);
            console.log('✅ Executed:', statement.split('(')[0].trim());
        } catch (error) {
            console.error('❌ Failed executing statement:', statement);
            throw error;
        }
    }
    console.log('✅ Operations registry tables ready');
}

if (require.main === module) {
    createOperationsRegistryTables()
        .then(() => process.exit(0))
        .catch(err => {
            console.error(err);
            process.exit(1);
        });
} else {
    module.exports = createOperationsRegistryTables;
}

