require('dotenv').config();
const mysql = require('mysql2/promise');

async function migrateFileUpload() {
    let connection;

    try {
        // Create database connection
        connection = await mysql.createConnection({
            host: process.env.DB_HOST || 'localhost',
            user: process.env.DB_USER || 'root',
            password: process.env.DB_PASSWORD || 'root',
            database: process.env.DB_NAME || 'hybrid_users',
            port: process.env.DB_PORT || 8889
        });

        console.log('Connected to database for migration');

        // Migration SQL statements
        const statements = [
            // Add file attachment columns to messages table
            `ALTER TABLE messages
            ADD COLUMN file_name VARCHAR(255) NULL COMMENT 'Original file name',
            ADD COLUMN file_size BIGINT NULL COMMENT 'File size in bytes',
            ADD COLUMN file_url VARCHAR(500) NULL COMMENT 'Public URL to access file',
            ADD COLUMN file_path VARCHAR(500) NULL COMMENT 'Server path to stored file',
            ADD COLUMN file_mime_type VARCHAR(100) NULL COMMENT 'MIME type of the file',
            ADD COLUMN file_thumbnail_path VARCHAR(500) NULL COMMENT 'Path to thumbnail for images'`,

            // Add index for file queries
            `CREATE INDEX idx_messages_file_type ON messages(message_type)`,
            `CREATE INDEX idx_messages_file_name ON messages(file_name)`
        ];

        // Execute each statement
        console.log('Running file upload migration...');
        for (const statement of statements) {
            try {
                await connection.execute(statement);
                console.log('Executed:', statement.split('\n')[0].trim());
            } catch (error) {
                // If columns already exist, that's fine
                if (error.code === 'ER_DUP_FIELDNAME') {
                    console.log('Columns already exist - skipping ALTER TABLE');
                } else if (error.code === 'ER_DUP_KEYNAME') {
                    console.log('Index already exists - skipping CREATE INDEX');
                } else {
                    throw error;
                }
            }
        }

        console.log('✅ File upload migration completed successfully!');

    } catch (error) {
        console.error('❌ Migration failed:', error);

        // If it's a "column already exists" error, that's fine - the migration was already run
        if (error.code === 'ER_DUP_FIELDNAME') {
            console.log('ℹ️  Migration columns already exist - skipping');
        } else {
            throw error;
        }
    } finally {
        if (connection) {
            await connection.end();
        }
    }
}

// Run the migration
migrateFileUpload().then(() => {
    console.log('Migration script completed!');
    process.exit(0);
}).catch((error) => {
    console.error('Migration script failed:', error);
    process.exit(1);
});
