require('dotenv').config({ path: require('path').resolve(__dirname, '../../../.env') });
const db = require('../config/db');

async function migrateApprovalTablesForPayloads() {
    try {
        console.log('🔄 Migrating approval tables to support payload-based workflow...');

        // Check current approval_requests table structure
        const [columns] = await db.execute(`
            SELECT COLUMN_NAME, COLUMN_TYPE, IS_NULLABLE, COLUMN_DEFAULT
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'approval_requests'
            ORDER BY ORDINAL_POSITION
        `);

        console.log('Current approval_requests columns:');
        columns.forEach(col => {
            console.log(`  ${col.COLUMN_NAME}: ${col.COLUMN_TYPE} ${col.IS_NULLABLE === 'YES' ? 'NULL' : 'NOT NULL'} ${col.COLUMN_DEFAULT ? `DEFAULT ${col.COLUMN_DEFAULT}` : ''}`);
        });

        // Add missing columns for payload-based workflow
        const alterations = [];

        // Check if payload column exists
        const hasPayload = columns.some(col => col.COLUMN_NAME === 'payload');
        if (!hasPayload) {
            alterations.push(`
                ALTER TABLE approval_requests
                ADD COLUMN payload JSON NULL COMMENT 'The full operation payload'
            `);
        }

        // Check if resource_ref column exists
        const hasResourceRef = columns.some(col => col.COLUMN_NAME === 'resource_ref');
        if (!hasResourceRef) {
            alterations.push(`
                ALTER TABLE approval_requests
                ADD COLUMN resource_ref VARCHAR(255) NULL COMMENT 'Client-visible correlation key'
            `);
        }

        // Check if correlation_id column exists
        const hasCorrelationId = columns.some(col => col.COLUMN_NAME === 'correlation_id');
        if (!hasCorrelationId) {
            alterations.push(`
                ALTER TABLE approval_requests
                ADD COLUMN correlation_id VARCHAR(255) NULL COMMENT 'For tracing across services'
            `);
        }

        // Check if current_step_order column exists
        const hasCurrentStepOrder = columns.some(col => col.COLUMN_NAME === 'current_step_order');
        if (!hasCurrentStepOrder) {
            alterations.push(`
                ALTER TABLE approval_requests
                ADD COLUMN current_step_order INT DEFAULT 1 COMMENT 'Current approval step order'
            `);
        }

        // Update status enum to include new statuses
        const statusColumn = columns.find(col => col.COLUMN_NAME === 'status');
        if (statusColumn && statusColumn.COLUMN_TYPE.includes("enum('pending','approved','rejected')")) {
            alterations.push(`
                ALTER TABLE approval_requests
                MODIFY COLUMN status ENUM('pending', 'in_progress', 'approved', 'changes_required', 'cancelled') DEFAULT 'pending'
            `);
        }

        // Add operation column if missing (for module.operation pattern)
        const hasOperation = columns.some(col => col.COLUMN_NAME === 'operation');
        if (!hasOperation) {
            alterations.push(`
                ALTER TABLE approval_requests
                ADD COLUMN operation VARCHAR(100) NULL COMMENT 'Operation type (create, update, delete, etc.)'
            `);
        }

        // Update requester_id to VARCHAR to support string IDs
        const requesterIdColumn = columns.find(col => col.COLUMN_NAME === 'requester_id');
        if (requesterIdColumn && requesterIdColumn.COLUMN_TYPE.startsWith('int')) {
            alterations.push(`
                ALTER TABLE approval_requests
                MODIFY COLUMN requester_id VARCHAR(100) NULL
            `);
        }

        // Add requester_name if missing
        const hasRequesterName = columns.some(col => col.COLUMN_NAME === 'requester_name');
        if (!hasRequesterName) {
            alterations.push(`
                ALTER TABLE approval_requests
                ADD COLUMN requester_name VARCHAR(255) NULL COMMENT 'Name of the requester'
            `);
        }

        // Execute alterations
        for (let i = 0; i < alterations.length; i++) {
            const alteration = alterations[i];
            console.log(`Executing alteration ${i + 1}/${alterations.length}:`);
            console.log(alteration.trim().substring(0, 100) + '...');

            try {
                await db.execute(alteration);
                console.log('✅ Alteration successful');
            } catch (error) {
                console.error('❌ Alteration failed:', error.message);
                // Continue with other alterations
            }
        }

        // Check if we need to create additional tables for the new workflow system
        console.log('\nChecking for additional workflow tables...');

        // Check if workflow_steps table exists
        const [workflowStepsTables] = await db.execute(`
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.TABLES
            WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'workflow_steps'
        `);

        if (workflowStepsTables.length === 0) {
            console.log('Creating workflow_steps table...');
            await db.execute(`
                CREATE TABLE workflow_steps (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    workflow_id INT NOT NULL,
                    step_order INT NOT NULL,
                    mode ENUM('serial', 'parallel') DEFAULT 'serial',
                    quorum INT DEFAULT 1,
                    assignees JSON NOT NULL,
                    sla_minutes INT,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    FOREIGN KEY (workflow_id) REFERENCES approval_workflows(id) ON DELETE CASCADE,
                    INDEX idx_workflow_order (workflow_id, step_order)
                )
            `);
            console.log('✅ workflow_steps table created');
        }

        // Check if workflow_policies table exists
        const [workflowPoliciesTables] = await db.execute(`
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.TABLES
            WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'workflow_policies'
        `);

        if (workflowPoliciesTables.length === 0) {
            console.log('Creating workflow_policies table...');
            await db.execute(`
                CREATE TABLE workflow_policies (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    module VARCHAR(100) NOT NULL,
                    operation VARCHAR(100) NOT NULL,
                    condition_expr TEXT,
                    workflow_id INT NOT NULL,
                    priority INT DEFAULT 1,
                    active BOOLEAN DEFAULT TRUE,
                    created_by VARCHAR(100),
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    FOREIGN KEY (workflow_id) REFERENCES approval_workflows(id) ON DELETE CASCADE,
                    INDEX idx_module_operation_priority (module, operation, priority DESC),
                    INDEX idx_active (active)
                )
            `);
            console.log('✅ workflow_policies table created');
        }

        // Check if approval_step_progress table exists
        const [stepProgressTables] = await db.execute(`
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.TABLES
            WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'approval_step_progress'
        `);

        if (stepProgressTables.length === 0) {
            console.log('Creating approval_step_progress table...');
            await db.execute(`
                CREATE TABLE approval_step_progress (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    request_id INT NOT NULL,
                    step_id INT NOT NULL,
                    status ENUM('pending', 'in_progress', 'completed', 'failed') DEFAULT 'pending',
                    started_at TIMESTAMP NULL,
                    completed_at TIMESTAMP NULL,
                    due_at TIMESTAMP NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (request_id) REFERENCES approval_requests(id) ON DELETE CASCADE,
                    FOREIGN KEY (step_id) REFERENCES workflow_steps(id) ON DELETE CASCADE,
                    INDEX idx_request_step (request_id, step_id),
                    INDEX idx_status (status)
                )
            `);
            console.log('✅ approval_step_progress table created');
        }

        // Update approval_actions table to support new workflow
        console.log('Updating approval_actions table...');
        const [actionsColumns] = await db.execute(`
            SELECT COLUMN_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'approval_actions'
        `);

        const actionsColumnNames = actionsColumns.map(col => col.COLUMN_NAME);

        if (!actionsColumnNames.includes('step_id')) {
            await db.execute(`
                ALTER TABLE approval_actions
                ADD COLUMN step_id INT NULL COMMENT 'Reference to workflow_steps'
            `);
            console.log('✅ Added step_id to approval_actions');
        }

        if (!actionsColumnNames.includes('action')) {
            await db.execute(`
                ALTER TABLE approval_actions
                MODIFY COLUMN action ENUM('approve','reject','delegate','escalate','comment') DEFAULT 'comment'
            `);
            console.log('✅ Updated action enum in approval_actions');
        }

        if (!actionsColumnNames.includes('metadata')) {
            await db.execute(`
                ALTER TABLE approval_actions
                ADD COLUMN metadata JSON NULL COMMENT 'Additional context'
            `);
            console.log('✅ Added metadata to approval_actions');
        }

        // Update approval_workflows to support new structure
        console.log('Updating approval_workflows table...');
        const [workflowColumns] = await db.execute(`
            SELECT COLUMN_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'approval_workflows'
        `);

        const workflowColumnNames = workflowColumns.map(col => col.COLUMN_NAME);

        if (!workflowColumnNames.includes('active')) {
            await db.execute(`
                ALTER TABLE approval_workflows
                ADD COLUMN active BOOLEAN DEFAULT TRUE COMMENT 'Whether workflow is active'
            `);
            console.log('✅ Added active column to approval_workflows');
        }

        console.log('✅ Migration completed successfully');

        // Show final table structures
        console.log('\nFinal table structures:');
        const tables = ['approval_requests', 'approval_workflows', 'approval_levels', 'approval_actions', 'workflow_steps', 'workflow_policies', 'approval_step_progress'];

        for (const table of tables) {
            const [tableExists] = await db.execute(`
                SELECT TABLE_NAME
                FROM INFORMATION_SCHEMA.TABLES
                WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?
            `, [table]);

            if (tableExists.length > 0) {
                const [tableColumns] = await db.execute(`
                    SELECT COLUMN_NAME, COLUMN_TYPE, IS_NULLABLE
                    FROM INFORMATION_SCHEMA.COLUMNS
                    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?
                    ORDER BY ORDINAL_POSITION
                `, [table]);

                console.log(`\n${table}:`);
                tableColumns.forEach(col => {
                    console.log(`  ${col.COLUMN_NAME}: ${col.COLUMN_TYPE} ${col.IS_NULLABLE === 'YES' ? 'NULL' : 'NOT NULL'}`);
                });
            } else {
                console.log(`\n${table}: NOT FOUND`);
            }
        }

    } catch (error) {
        console.error('❌ Error during migration:', error);
        throw error;
    }
}

// Run migration if executed directly
if (require.main === module) {
    migrateApprovalTablesForPayloads()
        .then(() => {
            console.log('\n🎉 Migration completed successfully!');
            process.exit(0);
        })
        .catch(err => {
            console.error('Migration failed:', err);
            process.exit(1);
        });
} else {
    module.exports = migrateApprovalTablesForPayloads;
}
