require('dotenv').config({ path: require('path').resolve(__dirname, '../../../.env') });
const db = require('../config/db');

async function seedCentralWorkflowData() {
    try {
        console.log('Seeding central workflow data...');
        // Check database connection
        const [dbTest] = await db.execute('SELECT DATABASE() as db_name');
        console.log('Connected to database:', dbTest[0].db_name);

        // Create a sample workflow for purchase orders
        const [workflowResult] = await db.execute(
            `INSERT INTO workflows (name, module, operation, description, active, created_by)
             VALUES (?, ?, ?, ?, ?, ?)`,
            [
                'Purchase Order Approval',
                'supply_chain',
                'create_purchase_order',
                'Standard approval workflow for purchase orders',
                true,
                'system'
            ]
        );

        const workflowId = workflowResult.insertId;
        console.log(`Created workflow with ID: ${workflowId}`);

        // Create workflow steps (serial approval: Manager -> CFO)
        await db.execute(
            `INSERT INTO workflow_steps (workflow_id, step_order, mode, quorum, assignees, sla_minutes)
             VALUES (?, ?, ?, ?, ?, ?)`,
            [
                workflowId,
                1,
                'serial',
                1,
                JSON.stringify([
                    { type: 'role', refId: 'manager', name: 'Manager' },
                    { type: 'role', refId: 'cfo', name: 'Chief Financial Officer' }
                ]),
                1440 // 24 hours SLA
            ]
        );

        console.log('Created workflow steps');

        // Create policy for purchase orders > $5000
        await db.execute(
            `INSERT INTO workflow_policies (module, operation, condition_expr, workflow_id, priority, active, created_by)
             VALUES (?, ?, ?, ?, ?, ?, ?)`,
            [
                'supply_chain',
                'create_purchase_order',
                'total_amount > 5000',
                workflowId,
                1,
                true,
                'system'
            ]
        );

        console.log('Created workflow policy');

        // Seed sample operations and callbacks
        const sampleOperations = [
            {
                service: 'procurement-service',
                module: 'supply_chain',
                operation: 'create_purchase_order',
                method: 'POST',
                path: '/api/purchase-orders',
                version: 'v1',
                payloadSchema: {
                    type: 'object',
                    required: ['vendorId', 'items', 'total_amount', 'currency'],
                    properties: {
                        vendorId: { type: 'integer' },
                        items: { type: 'array', items: { type: 'object' } },
                        total_amount: { type: 'number' },
                        currency: { type: 'string' }
                    }
                },
                callback: {
                    success_url: 'https://procurement.example.com/hooks/workflow/success',
                    failure_url: 'https://procurement.example.com/hooks/workflow/failure',
                    webhook_secret: 'procurement-secret',
                    retry_policy: { max_attempts: 5, backoff_seconds: 60 }
                }
            },
            {
                service: 'inventory-service',
                module: 'inventory',
                operation: 'adjust_stock',
                method: 'POST',
                path: '/api/inventory/adjust',
                version: 'v1',
                payloadSchema: {
                    type: 'object',
                    required: ['items', 'store_id'],
                    properties: {
                        store_id: { type: 'integer' },
                        reason: { type: 'string' },
                        items: {
                            type: 'array',
                            items: {
                                type: 'object',
                                required: ['sku', 'quantity'],
                                properties: {
                                    sku: { type: 'string' },
                                    quantity: { type: 'number' }
                                }
                            }
                        }
                    }
                },
                callback: {
                    success_url: 'https://inventory.example.com/hooks/workflow/success',
                    failure_url: 'https://inventory.example.com/hooks/workflow/failure',
                    webhook_secret: 'inventory-secret',
                    retry_policy: { max_attempts: 3, backoff_seconds: 120 }
                }
            }
        ];

        for (const op of sampleOperations) {
            const payloadSchemaJson = JSON.stringify(op.payloadSchema);
            await db.execute(
                `INSERT INTO operations (service, module, operation, method, path, version, payload_schema, is_active, created_by)
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                 ON DUPLICATE KEY UPDATE
                    version = VALUES(version),
                    payload_schema = VALUES(payload_schema),
                    is_active = VALUES(is_active),
                    updated_at = CURRENT_TIMESTAMP`,
                [
                    op.service,
                    op.module,
                    op.operation,
                    op.method,
                    op.path,
                    op.version,
                    payloadSchemaJson,
                    true,
                    'system'
                ]
            );

            const [operationRows] = await db.execute(
                `SELECT id FROM operations WHERE service = ? AND module = ? AND operation = ? AND method = ? AND path = ? LIMIT 1`,
                [op.service, op.module, op.operation, op.method, op.path]
            );

            if (!operationRows.length) {
                throw new Error(`Failed to fetch operation ID for ${op.service}:${op.operation}`);
            }

            const operationId = operationRows[0].id;

            if (op.callback) {
                await db.execute(
                    `INSERT INTO operation_callbacks (operation_id, success_url, failure_url, webhook_secret, retry_policy)
                     VALUES (?, ?, ?, ?, ?)
                     ON DUPLICATE KEY UPDATE
                        success_url = VALUES(success_url),
                        failure_url = VALUES(failure_url),
                        webhook_secret = VALUES(webhook_secret),
                        retry_policy = VALUES(retry_policy),
                        updated_at = CURRENT_TIMESTAMP`,
                    [
                        operationId,
                        op.callback.success_url,
                        op.callback.failure_url,
                        op.callback.webhook_secret,
                        JSON.stringify(op.callback.retry_policy)
                    ]
                );
            }

            console.log(`Seeded operation ${op.service}:${op.operation} (ID ${operationId})`);
        }

        console.log('✅ Central workflow data seeded successfully');

        // Display created data
        const [workflows] = await db.execute('SELECT * FROM workflows WHERE id = ?', [workflowId]);
        const [steps] = await db.execute('SELECT * FROM workflow_steps WHERE workflow_id = ?', [workflowId]);
        const [policies] = await db.execute('SELECT * FROM workflow_policies WHERE workflow_id = ?', [workflowId]);

        console.log('\nCreated workflow:', workflows[0]);
        console.log('Steps:', steps.map(s => {
            try {
                return { ...s, assignees: JSON.parse(s.assignees) };
            } catch (e) {
                return { ...s, assignees: s.assignees };
            }
        }));
        console.log('Policies:', policies);

    } catch (error) {
        console.error('❌ Error seeding central workflow data:', error);
        throw error;
    }
}

// If run directly, execute and exit
if (require.main === module) {
    seedCentralWorkflowData()
        .then(() => process.exit(0))
        .catch(err => {
            console.error(err);
            process.exit(1);
        });
} else {
    module.exports = seedCentralWorkflowData;
}
