const db = require('../config/db');

async function getModuleIdByName(moduleName) {
    // Try to find existing module by name
    const [rows] = await db.execute('SELECT id FROM modules WHERE name = ?', [moduleName]);
    if (rows.length > 0) {
        return rows[0].id;
    }

    // Determine if modules table has a description column
    const [columns] = await db.execute("SHOW COLUMNS FROM modules LIKE 'description'");
    if (columns.length > 0) {
        const [result] = await db.execute('INSERT INTO modules (name, description) VALUES (?, ?)', [moduleName, `${moduleName} module`]);
        return result.insertId;
    }

    // Fallback: insert with name only
    const [result] = await db.execute('INSERT INTO modules (name) VALUES (?)', [moduleName]);
    return result.insertId;
}

function buildSupplyChainPermissions() {
    // One permission per endpoint/action for fine-grained RBAC
    const perms = [
        // Raw Materials
        { name: 'SUPPLY_CHAIN_RAW_MATERIALS_CREATE', description: 'Create raw material' },
        { name: 'SUPPLY_CHAIN_RAW_MATERIALS_READ', description: 'View raw materials' },
        { name: 'SUPPLY_CHAIN_RAW_MATERIALS_UPDATE', description: 'Update raw material' },
        { name: 'SUPPLY_CHAIN_RAW_MATERIALS_UPDATE_PRICE', description: 'Update raw material price' },
        { name: 'SUPPLY_CHAIN_RAW_MATERIALS_DELETE', description: 'Delete raw material' },

        // Suppliers
        { name: 'SUPPLY_CHAIN_SUPPLIERS_CREATE', description: 'Create supplier' },
        { name: 'SUPPLY_CHAIN_SUPPLIERS_READ', description: 'View suppliers' },
        { name: 'SUPPLY_CHAIN_SUPPLIERS_UPDATE', description: 'Update supplier' },

        // Supplier Bank Accounts
        { name: 'SUPPLY_CHAIN_SUPPLIER_BANKS_CREATE', description: 'Create supplier bank account' },
        { name: 'SUPPLY_CHAIN_SUPPLIER_BANKS_READ', description: 'View supplier bank accounts' },
        { name: 'SUPPLY_CHAIN_SUPPLIER_BANKS_SET_PRIMARY', description: 'Set primary supplier bank account' },

        // Purchase Orders
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_CREATE', description: 'Create purchase order' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_READ', description: 'List purchase orders' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_READ_ONE', description: 'View purchase order details' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_UPDATE', description: 'Update purchase order' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_UPDATE_COSTS', description: 'Update purchase order costs' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_UPDATE_OFFLOADING_COST', description: 'Update purchase order offloading cost' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_UPDATE_GNC_ANALYSIS_COST', description: 'Update purchase order GNC analysis cost' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_UPDATE_STAGE', description: 'Update purchase order stage' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_DELETE', description: 'Delete purchase order' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_VIEW_PRINT_DATA', description: 'View purchase order print data' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_RECORD_PAYMENT', description: 'Record purchase order payment' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDERS_VIEW_PAYMENTS', description: 'View purchase order payments' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDER_ITEMS_ADD', description: 'Add items to purchase order' },
        { name: 'SUPPLY_CHAIN_PURCHASE_ORDER_VEHICLES_ADD', description: 'Add vehicle to purchase order' },

        // Weighbridge
        { name: 'SUPPLY_CHAIN_WEIGHBRIDGE_VIEW_MY_RECORDS', description: 'View own weighbridge records' },
        { name: 'SUPPLY_CHAIN_WEIGHBRIDGE_VIEW_RECORDS', description: 'View weighbridge records' },
        { name: 'SUPPLY_CHAIN_WEIGHBRIDGE_VIEW_PO_ITEMS', description: 'View PO items for weighbridge' },
        { name: 'SUPPLY_CHAIN_WEIGHBRIDGE_CREATE_RECORD', description: 'Create weighbridge record' },
        { name: 'SUPPLY_CHAIN_WEIGHBRIDGE_VIEW_RECORDS_BY_PO', description: 'View weighbridge records for PO' },
        { name: 'SUPPLY_CHAIN_WEIGHBRIDGE_VIEW_AVAILABLE_VEHICLES', description: 'View vehicles available for weighbridge' },
        { name: 'SUPPLY_CHAIN_WEIGHBRIDGE_UPDATE_RECORD', description: 'Update weighbridge record' },
        { name: 'SUPPLY_CHAIN_WEIGHBRIDGE_COMPLETE_PROCESS', description: 'Complete weighbridge process' },
        { name: 'SUPPLY_CHAIN_WEIGHBRIDGE_DELETE_RECORD', description: 'Delete weighbridge record' },

        // Gate
        { name: 'SUPPLY_CHAIN_GATE_SEARCH_PURCHASE_ORDERS', description: 'Search purchase orders at gate' },
        { name: 'SUPPLY_CHAIN_GATE_REGISTER_ARRIVAL', description: 'Register arrival at gate' },
        { name: 'SUPPLY_CHAIN_GATE_VIEW_VEHICLE_INFORMATION', description: 'View vehicle information for PO' },
        { name: 'SUPPLY_CHAIN_GATE_VIEW_PURCHASE_ORDER_ITEMS', description: 'View PO items at gate' },
        { name: 'SUPPLY_CHAIN_GATE_UPDATE_PURCHASE_ORDER_STAGE', description: 'Update PO stage at gate' },
        { name: 'SUPPLY_CHAIN_GATE_VIEW_PURCHASE_ORDERS_BY_STAGE', description: 'View POs by stage at gate' },

        // Quality
        { name: 'SUPPLY_CHAIN_QUALITY_VIEW_POS_BY_STAGE', description: 'View POs by stage for quality' },
        { name: 'SUPPLY_CHAIN_QUALITY_CREATE_INSPECTION', description: 'Create quality inspection' },
        { name: 'SUPPLY_CHAIN_QUALITY_VIEW_INSPECTIONS', description: 'View quality inspections for PO' },
        { name: 'SUPPLY_CHAIN_QUALITY_LIST_INSPECTIONS', description: 'List quality inspections' },
        { name: 'SUPPLY_CHAIN_QUALITY_UPDATE_INSPECTION', description: 'Update quality inspection' },
        { name: 'SUPPLY_CHAIN_QUALITY_COMPLETE_PURCHASE_ORDER', description: 'Complete PO in quality' },

        // Gate Clearances
        { name: 'SUPPLY_CHAIN_GATE_CLEARANCE_CREATE_OR_UPDATE', description: 'Create or update gate clearance' },
        { name: 'SUPPLY_CHAIN_GATE_CLEARANCE_VIEW_BY_ORDER', description: 'View gate clearance by order' },
        { name: 'SUPPLY_CHAIN_GATE_CLEARANCE_LIST', description: 'List all gate clearances' },
        { name: 'SUPPLY_CHAIN_GATE_CLEARANCE_DELETE', description: 'Delete gate clearance by order' },

        // Vehicle Registrations
        { name: 'SUPPLY_CHAIN_VEHICLE_REGISTRATION_CREATE', description: 'Register vehicle for delivery' },
        { name: 'SUPPLY_CHAIN_VEHICLE_REGISTRATION_LIST_BY_ORDER', description: 'List vehicle registrations by order' },
        { name: 'SUPPLY_CHAIN_VEHICLE_REGISTRATION_LIST', description: 'List vehicle registrations' },
        { name: 'SUPPLY_CHAIN_VEHICLE_REGISTRATION_UPDATE_STATUS', description: 'Update vehicle registration status' },
        { name: 'SUPPLY_CHAIN_VEHICLE_REGISTRATION_DELETE', description: 'Delete vehicle registration' },

        // Accounts (within Supply Chain)
        { name: 'SUPPLY_CHAIN_ACCOUNTS_VIEW_QC_COMPLETED_POS', description: 'View QC completed purchase orders' },
        { name: 'SUPPLY_CHAIN_ACCOUNTS_VIEW_PAYMENT_TRACKING', description: 'View payment tracking' },
        { name: 'SUPPLY_CHAIN_ACCOUNTS_CREATE_PAYMENT_REQUEST', description: 'Create payment request' },
        { name: 'SUPPLY_CHAIN_ACCOUNTS_UPDATE_PAYMENT_STATUS', description: 'Update payment status' },
        { name: 'SUPPLY_CHAIN_ACCOUNTS_APPROVE_PAYMENT', description: 'Approve payment' },

        // Reports
        { name: 'SUPPLY_CHAIN_REPORTS_DASHBOARD_VIEW_STATS', description: 'View dashboard stats report' },
        { name: 'SUPPLY_CHAIN_REPORTS_PURCHASE_ORDERS_VIEW', description: 'View purchase order reports' },
        { name: 'SUPPLY_CHAIN_REPORTS_PO_TRENDS_VIEW', description: 'View purchase order trends' },
        { name: 'SUPPLY_CHAIN_REPORTS_PO_STATUS_SUMMARY_VIEW', description: 'View purchase order status summary' },
        { name: 'SUPPLY_CHAIN_REPORTS_INVENTORY_VIEW', description: 'View inventory reports' },
        { name: 'SUPPLY_CHAIN_REPORTS_LOW_STOCK_ALERTS_VIEW', description: 'View low stock alerts' },
        { name: 'SUPPLY_CHAIN_REPORTS_STOCK_TURNOVER_VIEW', description: 'View stock turnover' },
        { name: 'SUPPLY_CHAIN_REPORTS_STOCK_MOVEMENTS_VIEW', description: 'View stock movements' },
        { name: 'SUPPLY_CHAIN_REPORTS_STOCK_ADJUSTMENTS_VIEW', description: 'View stock adjustments' },
        { name: 'SUPPLY_CHAIN_REPORTS_STOCK_ADJUSTMENTS_CREATE', description: 'Create stock adjustment' },
        { name: 'SUPPLY_CHAIN_REPORTS_SUPPLIER_PERFORMANCE_VIEW', description: 'View supplier performance reports' },
        { name: 'SUPPLY_CHAIN_REPORTS_QUALITY_VIEW', description: 'View quality reports' },
        { name: 'SUPPLY_CHAIN_REPORTS_QUALITY_TRENDS_VIEW', description: 'View quality trends' },
        { name: 'SUPPLY_CHAIN_REPORTS_DEFECT_ANALYSIS_VIEW', description: 'View defect analysis' },
        { name: 'SUPPLY_CHAIN_REPORTS_FINANCIAL_VIEW', description: 'View financial reports' },
        { name: 'SUPPLY_CHAIN_REPORTS_COST_ANALYSIS_VIEW', description: 'View cost analysis' },
        { name: 'SUPPLY_CHAIN_REPORTS_PAYMENT_STATUS_VIEW', description: 'View payment status reports' },
        { name: 'SUPPLY_CHAIN_REPORTS_BUDGET_VARIANCE_VIEW', description: 'View budget variance reports' },
        { name: 'SUPPLY_CHAIN_REPORTS_SUPPLIER_RISK_ASSESSMENT_VIEW', description: 'View supplier risk assessment reports' },
        { name: 'SUPPLY_CHAIN_REPORTS_WEIGHBRIDGE_VIEW', description: 'View weighbridge reports' },
        { name: 'SUPPLY_CHAIN_REPORTS_VEHICLE_UTILIZATION_VIEW', description: 'View vehicle utilization reports' },
        { name: 'SUPPLY_CHAIN_REPORTS_EXPORT', description: 'Export reports' },

        // Dashboard
        { name: 'SUPPLY_CHAIN_DASHBOARD_VIEW_STATS', description: 'View dashboard statistics' },
        { name: 'SUPPLY_CHAIN_DASHBOARD_VIEW_DAILY_TRUCKS', description: 'View daily trucks count' },
        { name: 'SUPPLY_CHAIN_DASHBOARD_VIEW_QUALITY_METRICS', description: 'View quality metrics' },
        { name: 'SUPPLY_CHAIN_DASHBOARD_VIEW_SUPPLIER_PERFORMANCE', description: 'View supplier performance metrics' },
        { name: 'SUPPLY_CHAIN_DASHBOARD_VIEW_FINANCIAL_METRICS', description: 'View financial metrics' },
        { name: 'SUPPLY_CHAIN_DASHBOARD_VIEW_WEIGHBRIDGE_ACTIVITY', description: 'View weighbridge activity' },

        // End of Day
        { name: 'SUPPLY_CHAIN_EOD_VIEW_HEALTH', description: 'View EOD health' },
        { name: 'SUPPLY_CHAIN_EOD_VIEW_DAILY_SUMMARY', description: 'View daily summary' },
        { name: 'SUPPLY_CHAIN_EOD_VIEW_TASKS', description: 'View EOD tasks and alerts' },
        { name: 'SUPPLY_CHAIN_EOD_MARK_TASK_COMPLETE', description: 'Mark EOD task complete' },
        { name: 'SUPPLY_CHAIN_EOD_VIEW_LEDGER', description: 'View EOD ledger data' },
        { name: 'SUPPLY_CHAIN_EOD_COMPLETE', description: 'Complete end of day process' },
        { name: 'SUPPLY_CHAIN_EOD_LOCK', description: 'Lock end of day data' },
        { name: 'SUPPLY_CHAIN_EOD_BACKUP_RECONCILE', description: 'Create EOD backup and reconcile' },
        { name: 'SUPPLY_CHAIN_EOD_EXPORT', description: 'Export EOD report' },
        { name: 'SUPPLY_CHAIN_EOD_VIEW_STATUS', description: 'View EOD status' },
        { name: 'SUPPLY_CHAIN_EOD_UNLOCK', description: 'Unlock EOD data' },
        { name: 'SUPPLY_CHAIN_EOD_VIEW_HISTORY', description: 'View EOD history' },
        { name: 'SUPPLY_CHAIN_EOD_VALIDATE', description: 'Validate EOD data' },
        { name: 'SUPPLY_CHAIN_EOD_VIEW_STATS', description: 'View EOD statistics' },
        { name: 'SUPPLY_CHAIN_EOD_VIEW_PENDING_ITEMS', description: 'View pending EOD items' },
        { name: 'SUPPLY_CHAIN_EOD_ADD_NOTE', description: 'Add note to EOD' },
        { name: 'SUPPLY_CHAIN_EOD_VIEW_NOTES', description: 'View EOD notes' },

        // SSE (optional)
        { name: 'SUPPLY_CHAIN_SSE_SUBSCRIBE_EVENTS', description: 'Subscribe to server-sent events' }
    ];

    return perms;
}

async function ensurePermission(moduleId, name, description) {
    const [existing] = await db.execute('SELECT id FROM permissions WHERE name = ?', [name]);
    if (existing.length > 0) {
        return existing[0].id;
    }

    // Detect table shape to include required columns if present
    const [cols] = await db.execute('SHOW COLUMNS FROM permissions');
    const colMap = Object.fromEntries(cols.map(c => [c.Field, c]));

    const fields = ['name', 'description', 'module_id'];
    const values = [name, description || name, moduleId];

    if (colMap.store_id) {
        fields.push('store_id');
        const defaultStoreId = process.env.DEFAULT_STORE_ID ? Number(process.env.DEFAULT_STORE_ID) : 1;
        values.push(defaultStoreId);
    }

    if (colMap.created_by) {
        fields.push('created_by');
        const createdBy = process.env.SEED_CREATED_BY ? Number(process.env.SEED_CREATED_BY) : 1;
        values.push(createdBy);
    }

    const placeholders = fields.map(() => '?').join(', ');
    const sql = `INSERT INTO permissions (${fields.join(', ')}) VALUES (${placeholders})`;
    const [result] = await db.execute(sql, values);
    return result.insertId;
}

async function seed() {
    try {
        console.log('Seeding Supply Chain permissions...');
        const moduleName = 'Supply Chain';
        const moduleId = await getModuleIdByName(moduleName);
        console.log(`Using module '${moduleName}' (id=${moduleId})`);

        const permissions = buildSupplyChainPermissions();
        let created = 0;
        let skipped = 0;
        for (const perm of permissions) {
            const [existing] = await db.execute('SELECT id FROM permissions WHERE name = ?', [perm.name]);
            if (existing.length > 0) {
                skipped++;
                continue;
            }
            await ensurePermission(moduleId, perm.name, perm.description);
            created++;
        }

        console.log(`Permissions created: ${created}, skipped (already present): ${skipped}`);
    } catch (err) {
        console.error('Error seeding Supply Chain permissions:', err);
    } finally {
        process.exit(0);
    }
}

if (require.main === module) {
    seed();
}

module.exports = { seed };