const db = require('../config/db');

// Supply Chain Approval Workflows Configuration
const supplyChainWorkflows = [
    // 1. Purchase Order Management Workflows
    {
        name: 'Purchase Order Creation',
        module: 'supply_chain',
        operation: 'create_purchase_order',
        description: 'Approval workflow for creating new purchase orders - requires approval for amounts over $10,000',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Manager Review',
            order: 1,
            required: true,
            roles: '["SUPPLY_CHAIN_MANAGER", "HEAD_OF_SUPPLY_CHAIN"]'
        },
        {
            name: 'Finance Approval',
            order: 2,
            required: true,
            roles: '["FINANCE_MANAGER", "CFO"]'
        }]
    },
    {
        name: 'Purchase Order Updates',
        module: 'supply_chain',
        operation: 'update_purchase_order',
        description: 'Approval workflow for updating existing purchase orders - requires approval for significant changes',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Manager Review',
            order: 1,
            required: true,
            roles: '["SUPPLY_CHAIN_MANAGER", "HEAD_OF_SUPPLY_CHAIN"]'
        }]
    },
    {
        name: 'Purchase Order Deletion',
        module: 'supply_chain',
        operation: 'delete_purchase_order',
        description: 'Approval workflow for deleting purchase orders - requires approval for audit trail',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Manager Approval',
            order: 1,
            required: true,
            roles: '["SUPPLY_CHAIN_MANAGER", "HEAD_OF_SUPPLY_CHAIN"]'
        },
        {
            name: 'Finance Approval',
            order: 2,
            required: true,
            roles: '["FINANCE_MANAGER", "CFO"]'
        }]
    },

    // 2. Supplier Management Workflows
    {
        name: 'Supplier Creation',
        module: 'supply_chain',
        operation: 'create_supplier',
        description: 'Approval workflow for creating new suppliers - requires approval for vendor management',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Procurement Review',
            order: 1,
            required: true,
            roles: '["PROCUREMENT_MANAGER", "SUPPLY_CHAIN_MANAGER"]'
        },
        {
            name: 'Legal Review',
            order: 2,
            required: true,
            roles: '["LEGAL_MANAGER", "COMPLIANCE_OFFICER"]'
        }]
    },
    {
        name: 'Supplier Updates',
        module: 'supply_chain',
        operation: 'update_supplier',
        description: 'Approval workflow for updating supplier information - requires approval for data integrity',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Procurement Review',
            order: 1,
            required: true,
            roles: '["PROCUREMENT_MANAGER", "SUPPLY_CHAIN_MANAGER"]'
        }]
    },
    {
        name: 'Supplier Deletion',
        module: 'supply_chain',
        operation: 'delete_supplier',
        description: 'Approval workflow for deleting suppliers - requires approval for audit trail',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Manager Approval',
            order: 1,
            required: true,
            roles: '["SUPPLY_CHAIN_MANAGER", "HEAD_OF_SUPPLY_CHAIN"]'
        },
        {
            name: 'Finance Review',
            order: 2,
            required: true,
            roles: '["FINANCE_MANAGER"]'
        }]
    },

    // 3. Raw Material Management Workflows
    {
        name: 'Raw Material Deletion',
        module: 'supply_chain',
        operation: 'delete_raw_material',
        description: 'Approval workflow for deleting raw materials - requires approval for inventory control',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Inventory Manager',
            order: 1,
            required: true,
            roles: '["INVENTORY_MANAGER", "SUPPLY_CHAIN_MANAGER"]'
        }]
    },

    // 4. Vehicle Management Workflows
    {
        name: 'Vehicle Deletion',
        module: 'supply_chain',
        operation: 'delete_vehicle',
        description: 'Approval workflow for deleting vehicles - requires approval for fleet management',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Fleet Manager',
            order: 1,
            required: true,
            roles: '["FLEET_MANAGER", "LOGISTICS_MANAGER"]'
        }]
    },

    // 5. Weighbridge Operations Workflows
    {
        name: 'Weighbridge Record Updates',
        module: 'supply_chain',
        operation: 'update_weighbridge_record',
        description: 'Approval workflow for updating weighbridge records - requires approval for data integrity',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Weighbridge Supervisor',
            order: 1,
            required: true,
            roles: '["WEIGHBRIDGE_SUPERVISOR", "QUALITY_MANAGER"]'
        }]
    },

    // 6. Quality Control Workflows
    {
        name: 'Quality Inspection Updates',
        module: 'supply_chain',
        operation: 'update_quality_inspection',
        description: 'Approval workflow for updating quality inspections - requires approval for data integrity',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Quality Manager',
            order: 1,
            required: true,
            roles: '["QUALITY_MANAGER", "HEAD_OF_QUALITY"]'
        }]
    },

    // 7. Financial Operations Workflows
    {
        name: 'Payment Processing',
        module: 'supply_chain',
        operation: 'create_payment',
        description: 'Approval workflow for processing payments - requires approval for financial control',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Accounts Manager',
            order: 1,
            required: true,
            roles: '["ACCOUNTS_MANAGER", "FINANCE_MANAGER"]'
        },
        {
            name: 'Finance Director',
            order: 2,
            required: true,
            roles: '["FINANCE_DIRECTOR", "CFO"]'
        }]
    },

    // 8. Cost Management Workflows
    {
        name: 'Cost Updates',
        module: 'supply_chain',
        operation: 'update_purchase_order_costs',
        description: 'Approval workflow for updating purchase order costs - requires approval for budget control',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Cost Controller',
            order: 1,
            required: true,
            roles: '["COST_CONTROLLER", "FINANCE_MANAGER"]'
        }]
    },

    // 9. Stage Progression Workflows
    {
        name: 'Purchase Order Stage Progression',
        module: 'supply_chain',
        operation: 'update_purchase_order_stage',
        description: 'Approval workflow for advancing purchase order stages - requires approval for process control',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Process Manager',
            order: 1,
            required: true,
            roles: '["PROCESS_MANAGER", "SUPPLY_CHAIN_MANAGER"]'
        }]
    },

    // 10. End of Day Operations Workflows
    {
        name: 'End of Day Completion',
        module: 'supply_chain',
        operation: 'complete_end_of_day',
        description: 'Approval workflow for completing end of day processes - requires approval for data integrity',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Operations Manager',
            order: 1,
            required: true,
            roles: '["OPERATIONS_MANAGER", "SUPPLY_CHAIN_MANAGER"]'
        },
        {
            name: 'Finance Review',
            order: 2,
            required: true,
            roles: '["FINANCE_MANAGER", "ACCOUNTS_MANAGER"]'
        }]
    },

    // 11. Report Export Workflows
    {
        name: 'Report Export',
        module: 'supply_chain',
        operation: 'export_reports',
        description: 'Approval workflow for exporting sensitive reports - requires approval for data security',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Data Security Officer',
            order: 1,
            required: true,
            roles: '["DATA_SECURITY_OFFICER", "IT_MANAGER"]'
        }]
    },

    // 12. Gate Clearance Workflows
    {
        name: 'Gate Clearance Management',
        module: 'supply_chain',
        operation: 'manage_gate_clearance',
        description: 'Approval workflow for gate clearance operations - requires approval for security control',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Security Manager',
            order: 1,
            required: true,
            roles: '["SECURITY_MANAGER", "FACILITY_MANAGER"]'
        }]
    },

    // 13. Vehicle Registration Workflows
    {
        name: 'Vehicle Registration Management',
        module: 'supply_chain',
        operation: 'manage_vehicle_registration',
        description: 'Approval workflow for vehicle registration operations - requires approval for fleet control',
        auto_approve: false,
        sequential_approval: true,
        status: 'active',
        created_by: 1, // Default admin user ID
        approval_levels: [{
            name: 'Fleet Manager',
            order: 1,
            required: true,
            roles: '["FLEET_MANAGER", "LOGISTICS_MANAGER"]'
        }]
    }
];

async function seedSupplyChainWorkflows() {
    try {
        console.log('🚀 Starting Supply Chain workflow seeding...');

        let createdWorkflows = 0;
        let skippedWorkflows = 0;
        const errors = [];

        for (const workflow of supplyChainWorkflows) {
            try {
                // Check if workflow already exists
                const [existing] = await db.query(`
                    SELECT id FROM approval_workflows 
                    WHERE name = ? AND module = ? AND operation = ?
                `, [workflow.name, workflow.module, workflow.operation]);

                if (existing.length > 0) {
                    console.log(`⚠️  Workflow "${workflow.name}" already exists, skipping...`);
                    skippedWorkflows++;
                    continue;
                }

                // Create workflow
                const [workflowResult] = await db.query(`
                    INSERT INTO approval_workflows 
                    (name, module, operation, description, auto_approve, sequential_approval, status, created_by, created_at, updated_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
                `, [
                    workflow.name,
                    workflow.module,
                    workflow.operation,
                    workflow.description,
                    workflow.auto_approve,
                    workflow.sequential_approval,
                    workflow.status,
                    workflow.created_by
                ]);

                const workflowId = workflowResult.insertId;

                // Create approval levels
                for (const level of workflow.approval_levels) {
                    await db.query(`
                        INSERT INTO approval_levels 
                        (workflow_id, name, \`order\`, required, roles, created_at, updated_at)
                        VALUES (?, ?, ?, ?, ?, NOW(), NOW())
                    `, [
                        workflowId,
                        level.name,
                        level.order,
                        level.required,
                        level.roles
                    ]);
                }

                console.log(`✅ Created workflow: ${workflow.name} (ID: ${workflowId})`);
                createdWorkflows++;

            } catch (error) {
                console.error(`❌ Error creating workflow "${workflow.name}":`, error.message);
                errors.push({
                    workflow: workflow.name,
                    error: error.message
                });
            }
        }

        console.log('\n📊 Supply Chain Workflow Seeding Summary:');
        console.log(`✅ Successfully created: ${createdWorkflows} workflows`);
        console.log(`⚠️  Skipped (already exist): ${skippedWorkflows} workflows`);

        if (errors.length > 0) {
            console.log(`❌ Failed to create: ${errors.length} workflows`);
            console.log('\n❌ Errors:');
            errors.forEach(error => {
                console.log(`  - ${error.workflow}: ${error.error}`);
            });
        }

        console.log('\n🎉 Supply Chain workflow seeding completed!');

    } catch (error) {
        console.error('❌ Error during workflow seeding:', error);
    } finally {
        process.exit(0);
    }
}

// Run the seeding if this script is executed directly
if (require.main === module) {
    seedSupplyChainWorkflows();
}

module.exports = { seedSupplyChainWorkflows };