const ChatModel = require('../models/chatModel');
const jwt = require('jsonwebtoken');

class ChatSocketService {
    constructor(io) {
        this.io = io;
        this.userSockets = new Map(); // userId -> socketId
        this.socketUsers = new Map(); // socketId -> userId

        this.io.use((socket, next) => {
            const token = socket.handshake.auth.token;
            if (!token) {
                return next(new Error('Authentication error'));
            }
            jwt.verify(token, process.env.AUTH_JWT_SECRET, (err, decoded) => {
                if (err) {
                    return next(new Error('Authentication error'));
                }
                socket.user = decoded.user;
                next();
            });
        });

        this.io.on('connection', this.handleConnection.bind(this));
    }

    handleConnection(socket) {
        console.log('New socket connection:', socket.id);

        // Authenticate user
        socket.on('authenticate', async(data) => {
            try {
                const { token } = data;
                if (!token) {
                    socket.emit('error', { message: 'Authentication token required' });
                    return;
                }

                const decoded = jwt.verify(token, process.env.JWT_SECRET);
                const userId = decoded.id;

                // Store user-socket mapping
                this.userSockets.set(userId, socket.id);
                this.socketUsers.set(socket.id, userId);

                // Join user to their personal room
                socket.join(`user_${userId}`);

                // Send authentication success
                socket.emit('authenticated', { userId });

                console.log(`User ${userId} authenticated on socket ${socket.id}`);
            } catch (error) {
                console.error('Socket authentication error:', error);
                socket.emit('error', { message: 'Authentication failed' });
            }
        });

        // Join conversation room
        socket.on('join_conversation', async(data) => {
            try {
                const { conversationId } = data;
                const userId = this.socketUsers.get(socket.id);

                if (!userId) {
                    socket.emit('error', { message: 'Not authenticated' });
                    return;
                }

                // Check if user is participant in conversation
                const isParticipant = await ChatModel.isUserInConversation(userId, conversationId);
                if (!isParticipant) {
                    socket.emit('error', { message: 'Access denied to conversation' });
                    return;
                }

                // Join conversation room
                socket.join(`conversation_${conversationId}`);
                socket.emit('joined_conversation', { conversationId });

                console.log(`User ${userId} joined conversation ${conversationId}`);
            } catch (error) {
                console.error('Error joining conversation:', error);
                socket.emit('error', { message: 'Failed to join conversation' });
            }
        });

        // Leave conversation room
        socket.on('leave_conversation', (data) => {
            const { conversationId } = data;
            socket.leave(`conversation_${conversationId}`);
            socket.emit('left_conversation', { conversationId });
        });

        // Handle incoming message
        socket.on('send_message', async(data) => {
            try {
                const { conversationId, message, messageType = 'text' } = data;
                const userId = this.socketUsers.get(socket.id);

                if (!userId) {
                    socket.emit('error', { message: 'Not authenticated' });
                    return;
                }

                if (!message || !conversationId) {
                    socket.emit('error', { message: 'Message and conversationId are required' });
                    return;
                }

                // Check if user is participant in conversation
                const isParticipant = await ChatModel.isUserInConversation(userId, conversationId);
                if (!isParticipant) {
                    socket.emit('error', { message: 'Access denied to conversation' });
                    return;
                }

                // Save message to database
                const messageId = await ChatModel.saveMessage(conversationId, userId, message, messageType);

                // Get sender information
                const senderInfo = await ChatModel.getUserById(userId);

                // Broadcast message to all participants in the conversation
                const messageData = {
                    id: messageId,
                    conversationId,
                    sender_id: userId,
                    sender_name: senderInfo?.name || 'Unknown',
                    sender_username: senderInfo?.username || 'unknown',
                    message,
                    message_type: messageType,
                    created_at: new Date()
                };

                this.io.to(`conversation_${conversationId}`).emit('new_message', messageData);

                // Send confirmation back to the sender
                socket.emit('message_sent', {
                    id: messageId,
                    conversationId,
                    success: true
                });

                console.log(`Message sent in conversation ${conversationId} by user ${userId}`);
            } catch (error) {
                console.error('Error sending message:', error);
                socket.emit('error', { message: 'Failed to send message' });
            }
        });

        // Handle file message
        socket.on('send_file_message', async(data) => {
            try {
                const {
                    conversationId,
                    message,
                    fileName,
                    fileSize,
                    fileUrl,
                    filePath,
                    fileMimeType,
                    fileThumbnailPath,
                    fileThumbnailUrl
                } = data;
                const userId = this.socketUsers.get(socket.id);

                if (!userId) {
                    socket.emit('error', { message: 'Not authenticated' });
                    return;
                }

                if (!conversationId) {
                    socket.emit('error', { message: 'Conversation ID is required' });
                    return;
                }

                // Check if user is participant in conversation
                const isParticipant = await ChatModel.isUserInConversation(userId, conversationId);
                if (!isParticipant) {
                    socket.emit('error', { message: 'Access denied to conversation' });
                    return;
                }

                // Validate required fields
                if (!fileMimeType) {
                    socket.emit('error', { message: 'File MIME type is required' });
                    return;
                }

                if (!fileUrl) {
                    socket.emit('error', { message: 'File URL is required' });
                    return;
                }

                // Determine message type
                const messageType = fileMimeType.startsWith('image/') ? 'image' : 'file';

                // Save file message to database
                const messageId = await ChatModel.saveFileMessage({
                    conversationId,
                    senderId: userId,
                    message: message || '',
                    messageType,
                    fileName,
                    fileSize,
                    fileUrl,
                    filePath,
                    fileMimeType,
                    fileThumbnailPath
                });

                // Get sender information
                const senderInfo = await ChatModel.getUserById(userId);

                // Broadcast file message to all participants in the conversation
                const messageData = {
                    id: messageId,
                    conversationId,
                    sender_id: userId,
                    sender_name: senderInfo?.name || 'Unknown',
                    sender_username: senderInfo?.username || 'unknown',
                    message: message || '',
                    message_type: messageType,
                    file_name: fileName,
                    file_size: fileSize,
                    file_url: fileUrl,
                    file_path: filePath,
                    file_mime_type: fileMimeType,
                    file_thumbnail_path: fileThumbnailPath,
                    file_thumbnail_url: fileThumbnailUrl,
                    created_at: new Date()
                };

                this.io.to(`conversation_${conversationId}`).emit('new_message', messageData);

                // Send confirmation back to the sender
                socket.emit('file_message_sent', {
                    id: messageId,
                    conversationId,
                    success: true
                });

                console.log(`File message sent in conversation ${conversationId} by user ${userId}`);
            } catch (error) {
                console.error('Error sending file message:', error);
                console.error('Error details:', {
                    message: error.message,
                    stack: error.stack,
                    data: data
                });
                socket.emit('error', { 
                    message: error.message || 'Failed to send file message',
                    details: error.stack
                });
            }
        });

        // Handle marking messages as read
        socket.on('mark_as_read', async(data) => {
            try {
                const { conversationId } = data;
                const userId = this.socketUsers.get(socket.id);

                if (!userId) {
                    socket.emit('error', { message: 'Not authenticated' });
                    return;
                }

                // Mark messages as read in database
                await ChatModel.markMessagesAsRead(userId, conversationId);

                // Get all participants in the conversation
                const participants = await ChatModel.getConversationParticipants(conversationId);
                console.log(`Conversation ${conversationId} has ${participants.length} participants:`, participants.map(p => p.id));

                // Notify all participants (including those not in the room) that messages have been read
                participants.forEach(participant => {
                    if (participant.id !== userId) { // Don't send to the reader themselves
                        const participantSocketId = this.userSockets.get(participant.id);
                        console.log(`Participant ${participant.id} socket: ${participantSocketId}`);
                        if (participantSocketId) {
                            console.log(`Sending messages_read event to user ${participant.id} for conversation ${conversationId}`);
                            this.io.to(participantSocketId).emit('messages_read', {
                                conversationId,
                                readerId: userId,
                                timestamp: new Date()
                            });
                        } else {
                            console.log(`User ${participant.id} is not connected, skipping messages_read notification`);
                        }
                    } else {
                        console.log(`Skipping reader ${participant.id} themselves`);
                    }
                });

                console.log(`User ${userId} marked messages as read in conversation ${conversationId}`);
            } catch (error) {
                console.error('Error marking messages as read:', error);
                socket.emit('error', { message: 'Failed to mark messages as read' });
            }
        });

        // Handle user online status
        socket.on('user_online', async(data) => {
            try {
                const userId = this.socketUsers.get(socket.id);
                if (!userId) {
                    socket.emit('error', { message: 'Not authenticated' });
                    return;
                }

                // Update user status to online
                await this.updateUserStatus(userId, 'online');

                // Broadcast to all other users that this user is online
                socket.broadcast.emit('user_status_changed', {
                    userId,
                    status: 'online',
                    timestamp: new Date()
                });

                console.log(`User ${userId} is now online`);
            } catch (error) {
                console.error('Error updating user online status:', error);
            }
        });

        // Handle user offline status
        socket.on('user_offline', async(data) => {
            try {
                const userId = this.socketUsers.get(socket.id);
                if (!userId) {
                    socket.emit('error', { message: 'Not authenticated' });
                    return;
                }

                // Update user status to offline
                await this.updateUserStatus(userId, 'offline');

                // Broadcast to all other users that this user is offline
                socket.broadcast.emit('user_status_changed', {
                    userId,
                    status: 'offline',
                    timestamp: new Date()
                });

                console.log(`User ${userId} is now offline`);
            } catch (error) {
                console.error('Error updating user offline status:', error);
            }
        });

        // Handle typing indicators
        socket.on('typing_start', (data) => {
            const { conversationId } = data;
            const userId = this.socketUsers.get(socket.id);

            if (userId) {
                socket.to(`conversation_${conversationId}`).emit('user_typing', {
                    userId,
                    conversationId
                });
            }
        });

        socket.on('typing_stop', (data) => {
            const { conversationId } = data;
            const userId = this.socketUsers.get(socket.id);

            if (userId) {
                socket.to(`conversation_${conversationId}`).emit('user_stopped_typing', {
                    userId,
                    conversationId
                });
            }
        });

        // Handle disconnection
        socket.on('disconnect', async() => {
            const userId = this.socketUsers.get(socket.id);
            if (userId) {
                // Update user status to offline
                await this.updateUserStatus(userId, 'offline');

                // Remove socket mappings
                this.userSockets.delete(userId);
                this.socketUsers.delete(socket.id);

                // Broadcast to all other users that this user is offline
                socket.broadcast.emit('user_status_changed', {
                    userId,
                    status: 'offline',
                    timestamp: new Date()
                });

                console.log(`User ${userId} disconnected`);
            }
        });
    }

    // Update user status in database
    async updateUserStatus(userId, status) {
        try {
            const db = require('../config/db');

            if (status === 'online') {
                // Update user as online and set last_seen_at to null
                await db.execute(`
                    UPDATE users
                    SET is_online = TRUE, last_seen_at = NULL
                    WHERE id = ?
                `, [userId]);
            } else if (status === 'offline') {
                // Update user as offline and set last_seen_at to current timestamp
                await db.execute(`
                    UPDATE users
                    SET is_online = FALSE, last_seen_at = NOW()
                    WHERE id = ?
                `, [userId]);
            }

            console.log(`Updated user ${userId} status to: ${status}`);

        } catch (error) {
            console.error('Error updating user status:', error);
        }
    }

    async getUserInfo(userId) {
        try {
            const db = require('../config/db');
            const [rows] = await db.execute(
                'SELECT name, username FROM users WHERE id = ?', [userId]
            );
            return rows[0] || { name: 'Unknown User', username: 'unknown' };
        } catch (error) {
            console.error('Error getting user info:', error);
            return { name: 'Unknown User', username: 'unknown' };
        }
    }

    // Send notification to specific user
    sendNotificationToUser(userId, event, data) {
        const socketId = this.userSockets.get(userId);
        if (socketId) {
            this.io.to(socketId).emit(event, data);
        }
    }

    // Broadcast to all users
    broadcastToAll(event, data) {
        this.io.emit(event, data);
    }

    // Get online users count
    getOnlineUsersCount() {
        return this.userSockets.size;
    }

    // Check if user is online
    isUserOnline(userId) {
        return this.userSockets.has(userId);
    }
}

module.exports = ChatSocketService;