const OperationQueueService = require('./operationQueueService');
const db = require('../config/db');

class ExecutionEngine {
    constructor() {
        this.isRunning = false;
        this.pollingInterval = 5000; // 5 seconds
        this.pollingTimer = null;
    }

    // Start the execution engine
    start() {
        if (this.isRunning) {
            console.log('Execution engine is already running');
            return;
        }

        this.isRunning = true;
        this.pollForOperations();

        console.log('✅ Execution engine started');
    }

    // Stop the execution engine
    stop() {
        if (!this.isRunning) {
            console.log('Execution engine is not running');
            return;
        }

        this.isRunning = false;

        if (this.pollingTimer) {
            clearTimeout(this.pollingTimer);
            this.pollingTimer = null;
        }

        console.log('🛑 Execution engine stopped');
    }

    // Poll for pending operations
    async pollForOperations() {
        if (!this.isRunning) return;

        try {
            const pendingOperations = await OperationQueueService.getPendingOperations();

            if (pendingOperations.length > 0) {
                console.log(`🔍 Found ${pendingOperations.length} pending operations to execute`);

                for (const operation of pendingOperations) {
                    await this.processOperation(operation);
                }
            }
        } catch (error) {
            console.error('❌ Error polling for operations:', error);
        }

        // Schedule next poll
        this.pollingTimer = setTimeout(() => {
            this.pollForOperations();
        }, this.pollingInterval);
    }

    // Process a single operation
    async processOperation(operation) {
        try {
            console.log(`🚀 Executing operation ${operation.id}: ${operation.operation}`);

            const result = await OperationQueueService.executeOperation(operation.id);

            console.log(`✅ Operation ${operation.id} completed successfully`);

            // Notify the user about successful execution
            await this.notifyUser(operation, result);

        } catch (error) {
            console.error(`❌ Operation ${operation.id} failed:`, error.message);
        }
    }

    // Notify user about execution result
    async notifyUser(operation, result) {
        try {
            // Get approval request details
            const approvalRequest = await this.getApprovalRequest(operation.approval_request_id);

            if (approvalRequest) {
                // Send notification to requester
                await this.sendNotification({
                    user_id: approvalRequest.requester_id,
                    title: 'Operation Executed Successfully',
                    message: `Your ${operation.operation} has been executed successfully.`,
                    data: result
                });

                console.log(`📧 Notification sent to user ${approvalRequest.requester_id}`);
            }
        } catch (error) {
            console.error('Error notifying user:', error);
        }
    }

    // Get approval request details
    async getApprovalRequest(requestId) {
        try {
            const query = `
                SELECT * FROM approval_requests WHERE id = ?
            `;

            const [requests] = await db.query(query, [requestId]);
            return requests[0];
        } catch (error) {
            console.error('Error getting approval request:', error);
            return null;
        }
    }

    // Send notification (placeholder - can be extended)
    async sendNotification(notificationData) {
        try {
            // For now, just log the notification
            // In a real implementation, this could send email, push notification, etc.
            console.log('📧 Notification:', notificationData);

            // You could integrate with your notification system here
            // await emailService.sendEmail(notificationData);
            // await pushNotificationService.send(notificationData);

        } catch (error) {
            console.error('Error sending notification:', error);
        }
    }

    // Get engine status
    getStatus() {
        return {
            isRunning: this.isRunning,
            pollingInterval: this.pollingInterval,
            lastPoll: this.lastPollTime
        };
    }

    // Force execution of a specific operation
    async forceExecuteOperation(operationId) {
        try {
            const operation = await OperationQueueService.getOperationById(operationId);
            if (!operation) {
                throw new Error(`Operation ${operationId} not found`);
            }

            console.log(`🔧 Force executing operation ${operationId}`);
            await this.processOperation(operation);

        } catch (error) {
            console.error(`Error force executing operation ${operationId}:`, error);
            throw error;
        }
    }
}

module.exports = new ExecutionEngine();