const fs = require('fs');
const path = require('path');
const crypto = require('crypto');
const sharp = require('sharp');

class FileUploadService {
    constructor() {
        this.uploadDir = path.join(__dirname, '..', 'uploads');
        this.thumbnailsDir = path.join(this.uploadDir, 'thumbnails');

        // Ensure directories exist
        this.ensureDirectories();
    }

    ensureDirectories() {
        const dirs = [this.uploadDir, this.thumbnailsDir];

        dirs.forEach(dir => {
            if (!fs.existsSync(dir)) {
                fs.mkdirSync(dir, { recursive: true });
                console.log(`Created directory: ${dir}`);
            }
        });
    }

    generateFileName(originalName) {
        const ext = path.extname(originalName);
        const baseName = path.basename(originalName, ext);
        const timestamp = Date.now();
        const random = crypto.randomBytes(8).toString('hex');
        return `${baseName}_${timestamp}_${random}${ext}`;
    }

    getFileType(mimeType) {
        if (mimeType.startsWith('image/')) return 'image';
        if (mimeType.startsWith('video/')) return 'video';
        if (mimeType.startsWith('audio/')) return 'audio';
        return 'file';
    }

    async saveFile(fileBuffer, originalName, mimeType) {
        try {
            const fileName = this.generateFileName(originalName);
            const filePath = path.join(this.uploadDir, fileName);
            const fileUrl = `/files/${fileName}`;

            // Write file to disk
            await fs.promises.writeFile(filePath, fileBuffer);

            const fileInfo = {
                fileName: originalName,
                filePath,
                fileUrl,
                fileSize: fileBuffer.length,
                mimeType,
                messageType: this.getFileType(mimeType)
            };

            // Generate thumbnail for images
            if (mimeType.startsWith('image/')) {
                const thumbnailPath = await this.generateThumbnail(fileBuffer, fileName);
                if (thumbnailPath) {
                    fileInfo.thumbnailPath = thumbnailPath;
                    fileInfo.thumbnailUrl = `/files/thumbnails/${path.basename(thumbnailPath)}`;
                }
            }

            return fileInfo;
        } catch (error) {
            console.error('Error saving file:', error);
            throw new Error('Failed to save file');
        }
    }

    async generateThumbnail(imageBuffer, originalFileName) {
        try {
            const ext = path.extname(originalFileName);
            const baseName = path.basename(originalFileName, ext);
            const thumbnailFileName = `${baseName}_thumb.jpg`;
            const thumbnailPath = path.join(this.thumbnailsDir, thumbnailFileName);

            await sharp(imageBuffer)
                .resize(200, 200, {
                    fit: 'inside',
                    withoutEnlargement: true
                })
                .jpeg({ quality: 80 })
                .toFile(thumbnailPath);

            return thumbnailPath;
        } catch (error) {
            console.error('Error generating thumbnail:', error);
            return null;
        }
    }

    async deleteFile(filePath) {
        try {
            if (fs.existsSync(filePath)) {
                await fs.promises.unlink(filePath);
                console.log(`Deleted file: ${filePath}`);
            }
        } catch (error) {
            console.error('Error deleting file:', error);
        }
    }

    async deleteMessageFiles(message) {
        try {
            if (message.file_path) {
                await this.deleteFile(message.file_path);
            }
            if (message.file_thumbnail_path) {
                await this.deleteFile(message.file_thumbnail_path);
            }
        } catch (error) {
            console.error('Error deleting message files:', error);
        }
    }

    validateFile(file, maxSize = 10 * 1024 * 1024) { // 10MB default
        const errors = [];

        // Check file size
        if (file.size > maxSize) {
            errors.push(`File size exceeds maximum limit of ${maxSize / (1024 * 1024)}MB`);
        }

        // Check file type
        const allowedTypes = [
            'image/jpeg', 'image/png', 'image/gif', 'image/webp',
            'application/pdf', 'text/plain', 'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        ];

        if (!allowedTypes.includes(file.mimetype)) {
            errors.push('File type not allowed');
        }

        return {
            isValid: errors.length === 0,
            errors
        };
    }

    getFileIcon(mimeType) {
        if (mimeType.startsWith('image/')) return 'image';
        if (mimeType === 'application/pdf') return 'picture_as_pdf';
        if (mimeType.includes('word')) return 'description';
        if (mimeType.includes('excel') || mimeType.includes('spreadsheet')) return 'table_chart';
        if (mimeType === 'text/plain') return 'text_snippet';
        return 'attach_file';
    }

    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
}

module.exports = new FileUploadService();

