const db = require('../config/db');
const { getIo } = require('../socket');

async function createNotification({ userId, title, message, link = null, type = 'info', priority = 'normal', icon = null, sourceModule = null, payload = null }) {
    const [result] = await db.query(
        `INSERT INTO notifications (user_id, title, message, link, type, priority, icon, source_module, payload)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [userId, title, message, link, type, priority, icon, sourceModule, payload ? JSON.stringify(payload) : null]
    );

    const notification = {
        id: result.insertId,
        user_id: userId,
        title,
        message,
        link,
        type,
        priority,
        icon,
        source_module: sourceModule,
        payload,
        created_at: new Date(),
    };

    const io = getIo();
    if (io) {
        io.to(`user:${userId}`).emit('notification:new', notification);
        io.to(`user:${userId}`).emit('notification:unread_count', await getUnreadCount(userId));
    }

    return notification;
}

async function listNotifications(userId, { unreadOnly = false, limit = 20, offset = 0 } = {}) {
    const where = ['user_id = ?'];
    const params = [userId];
    if (unreadOnly) {
        where.push('read_at IS NULL');
    }
    const [rows] = await db.query(
        `SELECT id, user_id, title, message, link, type, priority, icon, source_module, payload, created_at, read_at
         FROM notifications
         WHERE ${where.join(' AND ')}
         ORDER BY created_at DESC
         LIMIT ? OFFSET ?`,
        [...params, Number(limit), Number(offset)]
    );
    return rows;
}

async function markRead(userId, id) {
    await db.query(
        `UPDATE notifications SET read_at = NOW() WHERE id = ? AND user_id = ?`,
        [id, userId]
    );
    return { success: true };
}

async function markAllRead(userId) {
    await db.query(
        `UPDATE notifications SET read_at = NOW() WHERE user_id = ? AND read_at IS NULL`,
        [userId]
    );
    return { success: true };
}

async function getUnreadCount(userId) {
    const [rows] = await db.query(
        `SELECT COUNT(*) as cnt FROM notifications WHERE user_id = ? AND read_at IS NULL`,
        [userId]
    );
    return rows[0]?.cnt || 0;
}

module.exports = {
    createNotification,
    listNotifications,
    markRead,
    markAllRead,
    getUnreadCount,
};












