const db = require('../config/db');
const axios = require('axios');

class OperationQueueService {
    // Queue an operation for later execution
    async queueOperation(operationData) {
        try {
            const query = `
                INSERT INTO pending_operations 
                (approval_request_id, module, operation, endpoint, method, payload, headers, max_retries)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            `;

            const [result] = await db.query(query, [
                operationData.approval_request_id,
                operationData.module,
                operationData.operation,
                operationData.endpoint,
                operationData.method,
                JSON.stringify(operationData.payload),
                JSON.stringify(operationData.headers),
                operationData.max_retries || 3
            ]);

            console.log(`Operation queued with ID: ${result.insertId}`);
            return result.insertId;
        } catch (error) {
            console.error('Error queuing operation:', error);
            throw error;
        }
    }

    // Get pending operations for execution
    async getPendingOperations() {
        try {
            const query = `
                SELECT po.*, ar.status as approval_status
                FROM pending_operations po
                JOIN approval_requests ar ON po.approval_request_id = ar.id
                WHERE po.status = 'pending' 
                AND ar.status = 'approved'
                ORDER BY po.created_at ASC
            `;

            const [operations] = await db.query(query);
            return operations;
        } catch (error) {
            console.error('Error getting pending operations:', error);
            throw error;
        }
    }

    // Get operation by ID
    async getOperationById(operationId) {
        try {
            const query = `
                SELECT * FROM pending_operations WHERE id = ?
            `;

            const [operations] = await db.query(query, [operationId]);
            return operations[0];
        } catch (error) {
            console.error('Error getting operation by ID:', error);
            throw error;
        }
    }

    // Update operation status
    async updateOperationStatus(operationId, status, result = null, errorMessage = null) {
        try {
            const query = `
                UPDATE pending_operations 
                SET status = ?, execution_result = ?, error_message = ?, executed_at = ?
                WHERE id = ?
            `;

            const executedAt = status === 'completed' || status === 'failed' ? new Date() : null;

            await db.query(query, [
                status,
                result ? JSON.stringify(result) : null,
                errorMessage,
                executedAt,
                operationId
            ]);

            console.log(`Operation ${operationId} status updated to: ${status}`);
        } catch (error) {
            console.error('Error updating operation status:', error);
            throw error;
        }
    }

    // Update retry count
    async updateRetryCount(operationId, retryCount) {
        try {
            const query = `
                UPDATE pending_operations 
                SET retry_count = ?
                WHERE id = ?
            `;

            await db.query(query, [retryCount, operationId]);
        } catch (error) {
            console.error('Error updating retry count:', error);
            throw error;
        }
    }

    // Execute a queued operation
    async executeOperation(operationId) {
        try {
            console.log(`Executing operation ${operationId}...`);

            // 1. Get operation details
            const operation = await this.getOperationById(operationId);
            if (!operation) {
                throw new Error(`Operation ${operationId} not found`);
            }

            // 2. Update status to executing
            await this.updateOperationStatus(operationId, 'executing');

            // 3. Execute the operation
            const result = await this.callEndpoint(operation);

            // 4. Update as completed
            await this.updateOperationStatus(operationId, 'completed', result);

            console.log(`Operation ${operationId} completed successfully`);
            return result;
        } catch (error) {
            console.error(`Error executing operation ${operationId}:`, error);

            // 5. Handle failure and retry logic
            await this.handleExecutionError(operationId, error);
            throw error;
        }
    }

    // Call the actual endpoint
    async callEndpoint(operation) {
        try {
            const baseUrl = this.getModuleBaseUrl(operation.module);
            const url = `${baseUrl}${operation.endpoint}`;

            const config = {
                method: operation.method,
                url: url,
                headers: JSON.parse(operation.headers),
                data: operation.method !== 'GET' ? JSON.parse(operation.payload) : undefined,
                params: operation.method === 'GET' ? JSON.parse(operation.payload) : undefined,
                timeout: 30000 // 30 second timeout
            };

            console.log(`Calling endpoint: ${config.method} ${url}`);

            const response = await axios(config);
            return response.data;
        } catch (error) {
            console.error('Error calling endpoint:', error.response ? error.response.data : error.message);
            throw error;
        }
    }

    // Get module base URL
    getModuleBaseUrl(module) {
        const moduleUrls = {
            'supply_chain': 'http://localhost:3003',
            'finance': 'http://localhost:3004',
            'operations': 'http://localhost:3005',
            'accounts': 'http://localhost:3006'
        };

        return moduleUrls[module] || 'http://localhost:3000';
    }

    // Handle execution errors
    async handleExecutionError(operationId, error) {
        try {
            const operation = await this.getOperationById(operationId);

            if (operation.retry_count < operation.max_retries) {
                // Retry the operation
                console.log(`Retrying operation ${operationId} (attempt ${operation.retry_count + 1}/${operation.max_retries})`);
                await this.updateRetryCount(operationId, operation.retry_count + 1);
                await this.updateOperationStatus(operationId, 'pending');
            } else {
                // Mark as failed
                console.log(`Operation ${operationId} failed after ${operation.max_retries} retries`);
                await this.updateOperationStatus(operationId, 'failed', null, error.message);
            }
        } catch (updateError) {
            console.error('Error handling execution error:', updateError);
        }
    }

    // Get operations by approval request ID
    async getOperationsByApprovalRequest(approvalRequestId) {
        try {
            const query = `
                SELECT * FROM pending_operations 
                WHERE approval_request_id = ?
                ORDER BY created_at DESC
            `;

            const [operations] = await db.query(query, [approvalRequestId]);
            return operations;
        } catch (error) {
            console.error('Error getting operations by approval request:', error);
            throw error;
        }
    }
}

module.exports = new OperationQueueService();