require('dotenv').config();
const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');

async function setupChatTables() {
    let connection;

    try {
        // Create database connection
        connection = await mysql.createConnection({
            host: process.env.DB_HOST || 'localhost',
            user: process.env.DB_USER || 'root',
            password: process.env.DB_PASSWORD || '',
            database: process.env.DB_NAME || 'hybrid_erp'
        });

        console.log('Connected to database');

        // Read the SQL file
        const sqlFile = path.join(__dirname, 'scripts', 'create-chat-tables.sql');
        const sql = fs.readFileSync(sqlFile, 'utf8');

        // Split SQL into individual statements
        const statements = sql.split(';').filter(stmt => stmt.trim().length > 0);

        // Execute each statement
        for (const statement of statements) {
            if (statement.trim()) {
                await connection.execute(statement);
                console.log('Executed:', statement.trim().substring(0, 50) + '...');
            }
        }

        console.log('Chat tables created successfully!');

        // Test the tables by checking if they exist
        const [tables] = await connection.execute(
            "SHOW TABLES LIKE '%conversation%'"
        );

        console.log('Created tables:', tables.map(row => Object.values(row)[0]));

    } catch (error) {
        console.error('Error setting up chat tables:', error);
        process.exit(1);
    } finally {
        if (connection) {
            await connection.end();
        }
    }
}

// Run the setup
setupChatTables().then(() => {
    console.log('Chat setup completed successfully!');
    process.exit(0);
}).catch((error) => {
    console.error('Chat setup failed:', error);
    process.exit(1);
});

