const mysql = require('mysql2/promise');
require('dotenv').config({ path: require('path').resolve(__dirname, '../../.env') });

const dbConfig = {
    host: process.env.DB_HOST || '127.0.0.1',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || 'root',
    database: process.env.AUTH_DB_NAME || 'hybrid_users',
    port: process.env.DB_PORT || 3306
};

async function setupDatabase() {
    let connection;

    try {
        console.log('🔄 Setting up Auth database tables...');

        // First connect without specifying database to create it if needed
        const rootConfig = { ...dbConfig };
        delete rootConfig.database;

        connection = await mysql.createConnection(rootConfig);

        // Create database if it doesn't exist
        await connection.query(`CREATE DATABASE IF NOT EXISTS \`${dbConfig.database}\`;`);
        console.log(`✅ Database '${dbConfig.database}' created or already exists`);

        // Switch to the database
        await connection.query(`USE \`${dbConfig.database}\`;`);

        // Create tables individually
        const createStatements = [
            `CREATE TABLE IF NOT EXISTS modules (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(100) NOT NULL UNIQUE,
                description TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )`,

            `CREATE TABLE IF NOT EXISTS stores (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL,
                location VARCHAR(255),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )`,

            `CREATE TABLE IF NOT EXISTS roles (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(100) NOT NULL UNIQUE,
                description TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )`,

            `CREATE TABLE IF NOT EXISTS permissions (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(100) NOT NULL UNIQUE,
                description TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )`,

            `CREATE TABLE IF NOT EXISTS role_permissions (
                id INT AUTO_INCREMENT PRIMARY KEY,
                role_id INT NOT NULL,
                permission_id INT NOT NULL,
                FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE,
                FOREIGN KEY (permission_id) REFERENCES permissions(id) ON DELETE CASCADE,
                UNIQUE KEY unique_role_permission (role_id, permission_id)
            )`,

            `CREATE TABLE IF NOT EXISTS users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL,
                email VARCHAR(255) NOT NULL UNIQUE,
                username VARCHAR(100) UNIQUE,
                password VARCHAR(255) NOT NULL,
                phone_number VARCHAR(20),
                role_id INT,
                module_id INT,
                store_id INT,
                is_blocked BOOLEAN DEFAULT FALSE,
                is_active BOOLEAN DEFAULT TRUE,
                last_login TIMESTAMP NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )`,

            `CREATE TABLE IF NOT EXISTS user_permissions (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NOT NULL,
                permission_id INT NOT NULL,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
                FOREIGN KEY (permission_id) REFERENCES permissions(id) ON DELETE CASCADE,
                UNIQUE KEY unique_user_permission (user_id, permission_id)
            )`
        ];

        for (const statement of createStatements) {
            await connection.execute(statement);
        }

        // Insert default data
        const insertStatements = [
            `INSERT IGNORE INTO modules (name, description) VALUES
            ('accounts', 'Accounting and Finance Module'),
            ('supply_chain', 'Supply Chain Management'),
            ('dms', 'Document Management System'),
            ('hr', 'Human Resources'),
            ('auth', 'Authentication System')`,

            `INSERT IGNORE INTO stores (name, location) VALUES
            ('Main Store', 'Headquarters'),
            ('Branch Store', 'Downtown Branch')`,

            `INSERT IGNORE INTO roles (name, description) VALUES
            ('super_admin', 'Super Administrator with full access'),
            ('admin', 'Administrator'),
            ('manager', 'Manager'),
            ('cfo', 'Chief Financial Officer'),
            ('finance', 'Finance Officer'),
            ('user', 'Regular User'),
            ('viewer', 'Read-only User')`,

            `INSERT IGNORE INTO permissions (name, description) VALUES
            ('create_users', 'Can create new users'),
            ('edit_users', 'Can edit user information'),
            ('delete_users', 'Can delete users'),
            ('view_reports', 'Can view reports'),
            ('manage_inventory', 'Can manage inventory'),
            ('approve_documents', 'Can approve documents')`
        ];

        for (const statement of insertStatements) {
            await connection.execute(statement);
        }
        console.log('✅ All Auth database tables created successfully');

        // Insert sample user for testing
        const bcrypt = require('bcryptjs');
        const hashedPassword = await bcrypt.hash('bluegate', 10);

        const sampleUsers = [
            ['Super Admin', 'supp@css.com', 'supp', 1, 5, 1], // super_admin role, auth module, main store
            ['Accounts Admin', 'accounts@css.com', 'accounts', 2, 1, 1], // admin role, accounts module, main store
            ['Regular User', 'user@css.com', 'user', 4, 1, 1] // user role, accounts module, main store
        ];

        for (const [name, email, username, roleId, moduleId, storeId] of sampleUsers) {
            await connection.execute(
                'INSERT IGNORE INTO users (name, email, username, password, role_id, module_id, store_id) VALUES (?, ?, ?, ?, ?, ?, ?)',
                [name, email, username, hashedPassword, roleId, moduleId, storeId]
            );
        }

        console.log('✅ Sample users created successfully');
        console.log('🔐 Sample login credentials:');
        console.log('   Email: supp@css.com, Password: bluegate (Super Admin)');
        console.log('   Email: accounts@css.com, Password: bluegate (Accounts Admin)');
        console.log('   Email: user@css.com, Password: bluegate (Regular User)');

    } catch (error) {
        console.error('❌ Error setting up Auth database:', error);
        throw error;
    } finally {
        if (connection) {
            await connection.end();
        }
    }
}

// Run if called directly
if (require.main === module) {
    setupDatabase()
        .then(() => {
            console.log('🎉 Auth database setup completed!');
            process.exit(0);
        })
        .catch((error) => {
            console.error('💥 Auth database setup failed:', error);
            process.exit(1);
        });
}

module.exports = { setupDatabase };
